package org.evolizer.changedistiller.test.scchange;

import static org.junit.Assert.assertNotNull;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.evolizer.changedistiller.distilling.Distiller;
import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.ChangeType;
import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.changedistiller.model.entities.AttributeHistory;
import org.evolizer.changedistiller.model.entities.ClassHistory;
import org.evolizer.changedistiller.model.entities.MethodHistory;
import org.evolizer.changedistiller.model.entities.SourceCodeChange;
import org.evolizer.changedistiller.model.entities.StructureEntityVersion;
import org.evolizer.changedistiller.test.Activator;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

public class DifferenceJobTest extends AbstractDistillerTest {

    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
        TEST_FOLDER = "./test_data/integration/";
        UNIQUE_NAME = "test.Test";
        MODIFIERS = ChangeModifier.PUBLIC | ChangeModifier.FINAL;
        AbstractDistillerTest.setUpBeforeClass();

        fExpectedChanges = new ArrayList<SourceCodeChange>(28);

        fExpectedChanges.add(createUpdateOperation(
                ChangeType.DOC_UPDATE,
                "test.Test",
                EntityType.CLASS,
                ChangeModifier.PUBLIC | ChangeModifier.FINAL,
                "1.2",
                "/**\n" + " * This is our first (left) test class.\n" + " * @author Beat Fluri\n" + " */",
                EntityType.JAVADOC,
                0,
                15,
                69,
                "test.Test",
                EntityType.TYPE_DECLARATION,
                ChangeModifier.PUBLIC,
                15,
                1128,
                "/**\n" + " * This is our second (right) test class.\n" + " * @author Beat Fluri\n" + " */",
                EntityType.JAVADOC,
                0,
                15,
                71));
        fExpectedChanges.add(createInsertOperation(
                ChangeType.REMOVING_CLASS_DERIVABILITY,
                "test.Test",
                EntityType.CLASS,
                ChangeModifier.PUBLIC | ChangeModifier.FINAL,
                "1.2",
                "final",
                EntityType.MODIFIER,
                0,
                94,
                5,
                "",
                EntityType.MODIFIERS,
                0,
                85,
                6));
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.METHOD_RENAMING,
                "test.Test.Bar.newMethod()",
                EntityType.METHOD,
                ChangeModifier.PRIVATE,
                "1.2",
                "method",
                EntityType.METHOD_DECLARATION,
                ChangeModifier.PRIVATE,
                946,
                102,
                "test.Test.Bar",
                EntityType.CLASS,
                ChangeModifier.PRIVATE,
                924,
                127,
                "newMethod",
                EntityType.METHOD_DECLARATION,
                ChangeModifier.PRIVATE,
                839,
                107));
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.COMMENT_UPDATE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "// check if number is greater than -1",
                EntityType.LINE_COMMENT,
                0,
                431,
                37,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                289,
                594,
                "// check if number is greater than 0",
                EntityType.LINE_COMMENT,
                0,
                390,
                36));
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.COMMENT_UPDATE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "// check the huga number\n\t\t// and some new",
                EntityType.LINE_COMMENT,
                0,
                528,
                42,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                289,
                594,
                "// check the number",
                EntityType.LINE_COMMENT,
                0,
                488,
                19));
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.CONDITION_EXPRESSION_CHANGE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "check",
                EntityType.IF_STATEMENT,
                0,
                574,
                306,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                289,
                594,
                "!check",
                EntityType.IF_STATEMENT,
                0,
                513,
                215));
        fExpectedChanges.add(createInsertOperation(
                ChangeType.STATEMENT_INSERT,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "42",
                EntityType.RETURN_STATEMENT,
                0,
                731,
                10,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                289,
                594));
        fExpectedChanges.add(createMoveOperation(
                ChangeType.STATEMENT_PARENT_CHANGE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "b=Math.abs(number);",
                EntityType.ASSIGNMENT,
                0,
                817,
                20,
                567,
                20,
                "!check",
                EntityType.ELSE_STATEMENT,
                0,
                792,
                88,
                "!check",
                EntityType.THEN_STATEMENT,
                0,
                585,
                201));
        fExpectedChanges.add(createMoveOperation(
                ChangeType.STATEMENT_PARENT_CHANGE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "b=Math.round(Math.random());",
                EntityType.ASSIGNMENT,
                0,
                722,
                42,
                640,
                45,
                "!check",
                EntityType.THEN_STATEMENT,
                0,
                585,
                201,
                "!check",
                EntityType.ELSE_STATEMENT,
                0,
                792,
                88));
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.COMMENT_UPDATE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "/* mimimi */",
                EntityType.BLOCK_COMMENT,
                0,
                751,
                12,
                "!check",
                EntityType.ELSE_STATEMENT,
                0,
                792,
                88,
                "/* mimimi mi */",
                EntityType.BLOCK_COMMENT,
                0,
                669,
                15));
        fExpectedChanges.add(createMoveOperation(
                ChangeType.COMMENT_MOVE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "/* mimimi mi */",
                EntityType.BLOCK_COMMENT,
                0,
                751,
                12,
                669,
                15,
                "!check",
                EntityType.THEN_STATEMENT,
                0,
                585,
                201,
                "!check",
                EntityType.ELSE_STATEMENT,
                0,
                792,
                88));
        fExpectedChanges.add(createDeleteOperation(
                ChangeType.STATEMENT_DELETE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "System.out.println(\"left\");",
                EntityType.METHOD_INVOCATION,
                0,
                398,
                26,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                289,
                594));
        fExpectedChanges.add(createDeleteOperation(
                ChangeType.COMMENT_DELETE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "/* This is the most beautiful comment in the world\n" + "\t\t\t * and soon it will be gone :'(\n"
                        + "\t\t\t */",
                EntityType.BLOCK_COMMENT,
                0,
                590,
                92,
                "!check",
                EntityType.THEN_STATEMENT,
                0,
                585,
                201));
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.DOC_UPDATE,
                "test.Test.foo(int)",
                EntityType.METHOD,
                ChangeModifier.PROTECTED,
                "1.2",
                "/**\n" + "\t * Yet another method with a comment\n" + "\t * @param number\n" + "\t * @return\n"
                        + "\t */",
                EntityType.JAVADOC,
                0,
                289,
                76,
                "test.Test.foo(int)",
                EntityType.METHOD_DECLARATION,
                ChangeModifier.PUBLIC,
                289,
                594,
                "/**\n" + "\t * Yet another method with a better comment\n" + "\t * @param number\n" + "\t * @return\n"
                        + "\t */",
                EntityType.JAVADOC,
                0,
                268,
                83));
        fExpectedChanges.add(createInsertOperation(
                ChangeType.ADDITIONAL_FUNCTIONALITY,
                "test.Test",
                EntityType.CLASS,
                ChangeModifier.PUBLIC | ChangeModifier.FINAL,
                "1.2",
                "test.Test.newBar(long)",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                975,
                67,
                "test.Test",
                EntityType.CLASS,
                ChangeModifier.PUBLIC,
                15,
                1128));
        fExpectedChanges.add(createInsertOperation(
                ChangeType.ADDITIONAL_FUNCTIONALITY,
                "test.Test",
                EntityType.CLASS,
                ChangeModifier.PUBLIC | ChangeModifier.FINAL,
                "1.2",
                "test.Test.emptyMethod()",
                EntityType.METHOD,
                ChangeModifier.PUBLIC,
                747,
                29,
                "test.Test",
                EntityType.CLASS,
                ChangeModifier.PUBLIC,
                15,
                1128));
        SourceCodeChange scc =
                createDeleteOperation(
                        ChangeType.REMOVED_FUNCTIONALITY,
                        "test.Test",
                        EntityType.CLASS,
                        ChangeModifier.PUBLIC | ChangeModifier.FINAL,
                        "1.2",
                        "test.Test.bar(int)",
                        EntityType.METHOD,
                        ChangeModifier.PUBLIC,
                        1077,
                        63,
                        "test.Test",
                        EntityType.CLASS,
                        ChangeModifier.PUBLIC,
                        15,
                        1128);
        fExpectedChanges.add(scc);
        scc =
                createUpdateOperation(
                        ChangeType.ATTRIBUTE_RENAMING,
                        "test.Test.aNewString : String",
                        EntityType.ATTRIBUTE,
                        ChangeModifier.PUBLIC,
                        "1.2",
                        "aString",
                        EntityType.FIELD_DECLARATION,
                        ChangeModifier.PUBLIC,
                        108,
                        74,
                        "test.Test",
                        EntityType.CLASS,
                        ChangeModifier.PUBLIC,
                        15,
                        1128,
                        "aNewString",
                        EntityType.FIELD_DECLARATION,
                        ChangeModifier.PUBLIC,
                        116,
                        85);
        fExpectedChanges.add(scc);
        fExpectedChanges.add(createUpdateOperation(
                ChangeType.DOC_UPDATE,
                "test.Test.aNewString : String",
                EntityType.ATTRIBUTE,
                ChangeModifier.PUBLIC,
                "1.2",
                "/**\n" + "\t * Yet another attribute\n" + "\t */",
                EntityType.JAVADOC,
                0,
                108,
                34,
                "test.Test.aNewString : String",
                EntityType.FIELD_DECLARATION,
                ChangeModifier.PUBLIC,
                108,
                74,
                "/**\n" + "\t * Yet another changed attribute\n" + "\t */",
                EntityType.JAVADOC,
                0,
                116,
                42));
        scc =
                createDeleteOperation(
                        ChangeType.REMOVED_OBJECT_STATE,
                        "test.Test",
                        EntityType.CLASS,
                        ChangeModifier.PUBLIC | ChangeModifier.FINAL,
                        "1.2",
                        "test.Test.anInteger : Integer",
                        EntityType.ATTRIBUTE,
                        ChangeModifier.PUBLIC,
                        243,
                        42,
                        "test.Test",
                        EntityType.CLASS,
                        ChangeModifier.PUBLIC,
                        15,
                        1128);
        fExpectedChanges.add(scc);
        scc =
                createUpdateOperation(
                        ChangeType.DECREASING_ACCESSIBILITY_CHANGE,
                        "test.Test.foo(int)",
                        EntityType.METHOD,
                        ChangeModifier.PROTECTED,
                        "1.2",
                        "public",
                        EntityType.MODIFIER,
                        0,
                        367,
                        6,
                        "",
                        EntityType.MODIFIERS,
                        0,
                        353,
                        9,
                        "protected",
                        EntityType.MODIFIER,
                        0,
                        353,
                        9);
        fExpectedChanges.add(scc);
        scc =
                createUpdateOperation(
                        ChangeType.ATTRIBUTE_TYPE_CHANGE,
                        "test.Test.aField : int",
                        EntityType.ATTRIBUTE,
                        ChangeModifier.PUBLIC,
                        "1.2",
                        "String",
                        EntityType.SIMPLE_TYPE,
                        0,
                        225,
                        6,
                        "test.Test.aField : int",
                        EntityType.FIELD_DECLARATION,
                        ChangeModifier.PUBLIC,
                        218,
                        21,
                        "int",
                        EntityType.PRIMITIVE_TYPE,
                        0,
                        253,
                        3);
        fExpectedChanges.add(scc);
    }

    @AfterClass
    public static void tearDownAfterClass() throws Exception {
        AbstractDistillerTest.tearDownAfterClass();
    }

    @Test
    public void testSourceCodeChanges() throws Exception {
        checkChanges(new LinkedList<SourceCodeChange>(fDJob.getSourceCodeChanges()), 22, 0);
    }

    @Test
    public void testChangeClassification() throws Exception {
        List<SourceCodeChange> result = fDJob.getSourceCodeChanges();
        checkChanges(result, 22, 0);
    }

    @Test
    public void testClassHistory() throws Exception {
        IFile third = fFolder.getFile("1_3.java");
        third.create(Activator.openBundledFile(TEST_FOLDER + "TestThird.java"), true, null);

        ClassHistory ch = fDJob.getClassHistory();
        fDJob = new Distiller();
        fDJob.setClassHistory(ch);
        fDJob.performDistilling(fFolder.getFile("1_2.java"), third);

        ch.updateLatestVersionWithRevision(new Revision("1.3"));

        assertNotNull(ch);

        Assert.assertEquals(2, ch.getVersions().size());
        StructureEntityVersion sev = ch.getVersions().get(0);
        Assert.assertEquals("test.Test", sev.getUniqueName());
        Assert.assertEquals(EntityType.CLASS, sev.getType());
        Assert.assertEquals("1.2", sev.getRevision().getNumber());
        Assert.assertEquals(6, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.ADDITIONAL_FUNCTIONALITY));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.REMOVED_OBJECT_STATE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.REMOVED_FUNCTIONALITY));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.REMOVING_CLASS_DERIVABILITY));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DOC_UPDATE));
        sev = ch.getVersions().get(1);
        Assert.assertEquals("test.Test", sev.getUniqueName());
        Assert.assertEquals(EntityType.CLASS, sev.getType());
        Assert.assertEquals("1.3", sev.getRevision().getNumber());
        Assert.assertEquals(2, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.REMOVED_FUNCTIONALITY));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DOC_DELETE));

        // -- Attribute Histories --
        Assert.assertEquals(2, ch.getAttributeHistories().size());
        Iterator<AttributeHistory> ahs = ch.getAttributeHistories().values().iterator();

        // history for "test.Test.aString : String"
        AttributeHistory ah = ahs.next();
        Assert.assertEquals(2, ah.getVersions().size());
        sev = ah.getVersions().get(0);
        Assert.assertEquals("test.Test.aNewString : String", sev.getUniqueName());
        Assert.assertEquals(EntityType.ATTRIBUTE, sev.getType());
        Assert.assertEquals("1.2", sev.getRevision().getNumber());
        Assert.assertEquals(2, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.ATTRIBUTE_RENAMING));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DOC_UPDATE));
        sev = ah.getVersions().get(1);
        Assert.assertEquals("1.3", sev.getRevision().getNumber());
        Assert.assertEquals("test.Test.aNewString : String", sev.getUniqueName());
        Assert.assertEquals(EntityType.ATTRIBUTE, sev.getType());
        Assert.assertEquals(1, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DOC_UPDATE));

        // history for "test.Test.aField : String"
        ah = ahs.next();
        Assert.assertEquals(1, ah.getVersions().size());
        sev = ah.getVersions().get(0);
        Assert.assertEquals("test.Test.aField : int", sev.getUniqueName());
        Assert.assertEquals(EntityType.ATTRIBUTE, sev.getType());
        Assert.assertEquals("1.2", sev.getRevision().getNumber());
        Assert.assertEquals(1, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.ATTRIBUTE_TYPE_CHANGE));

        // -- Method Histories --
        Assert.assertEquals(1, ch.getMethodHistories().size());
        Iterator<MethodHistory> mhs = ch.getMethodHistories().values().iterator();

        // history for "test.Test.foo(int)"
        MethodHistory mh = mhs.next();
        Assert.assertEquals(2, mh.getVersions().size());
        // first version
        sev = mh.getVersions().get(0);
        Assert.assertEquals("test.Test.foo(int)", sev.getUniqueName());
        Assert.assertEquals(EntityType.METHOD, sev.getType());
        Assert.assertEquals("1.2", sev.getRevision().getNumber());
        Assert.assertEquals(12, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.STATEMENT_INSERT));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.STATEMENT_DELETE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.STATEMENT_PARENT_CHANGE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.STATEMENT_PARENT_CHANGE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.CONDITION_EXPRESSION_CHANGE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.COMMENT_DELETE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.COMMENT_MOVE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.COMMENT_UPDATE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.COMMENT_UPDATE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.COMMENT_UPDATE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DECREASING_ACCESSIBILITY_CHANGE));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DOC_UPDATE));
        // second version
        sev = mh.getVersions().get(1);
        Assert.assertEquals("test.Test.fooHuga(int)", sev.getUniqueName());
        Assert.assertEquals(EntityType.METHOD, sev.getType());
        Assert.assertEquals("1.3", sev.getRevision().getNumber());
        Assert.assertEquals(2, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.METHOD_RENAMING));
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.DOC_DELETE));

        // -- Inner Class History --
        Assert.assertEquals(1, ch.getInnerClassHistories().size());
        Iterator<ClassHistory> chs = ch.getInnerClassHistories().values().iterator();

        // history for "test.Test.Bar"
        ClassHistory ch2 = chs.next();
        Assert.assertEquals(2, ch2.getVersions().size());
        // first version
        sev = ch2.getVersions().get(0);
        Assert.assertEquals("test.Test.Bar", sev.getUniqueName());
        Assert.assertEquals(EntityType.CLASS, sev.getType());
        Assert.assertEquals("1.2", sev.getRevision().getNumber());
        Assert.assertTrue(sev.getSourceCodeChanges().isEmpty());
        // second version
        sev = ch2.getVersions().get(1);
        Assert.assertEquals("test.Test.Bar", sev.getUniqueName());
        Assert.assertEquals(EntityType.CLASS, sev.getType());
        Assert.assertEquals("1.3", sev.getRevision().getNumber());
        Assert.assertTrue(sev.getSourceCodeChanges().isEmpty());

        // -- Attribute Histories --
        Assert.assertTrue(ch2.getAttributeHistories().isEmpty());

        // -- Method Histories --
        Assert.assertEquals(1, ch2.getMethodHistories().size());
        mhs = ch2.getMethodHistories().values().iterator();

        // history for "test.Test.Bar.method()"
        mh = mhs.next();
        Assert.assertEquals(2, mh.getVersions().size());
        // first version
        sev = mh.getVersions().get(0);
        Assert.assertEquals("test.Test.Bar.newMethod()", sev.getUniqueName());
        Assert.assertEquals(EntityType.METHOD, sev.getType());
        Assert.assertEquals("1.2", sev.getRevision().getNumber());
        Assert.assertEquals(1, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.METHOD_RENAMING));
        // second version
        sev = mh.getVersions().get(1);
        Assert.assertEquals("test.Test.Bar.newMethod()", sev.getUniqueName());
        Assert.assertEquals(EntityType.METHOD, sev.getType());
        Assert.assertEquals("1.3", sev.getRevision().getNumber());
        Assert.assertEquals(1, sev.getSourceCodeChanges().size());
        Assert.assertTrue(containsChangeType(sev.getSourceCodeChanges(), ChangeType.STATEMENT_DELETE));
    }
}