package org.evolizer.changedistiller.test.scchange;

import java.util.List;

import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.ChangeType;
import org.evolizer.changedistiller.model.entities.SourceCodeChange;
import org.evolizer.changedistiller.model.entities.Delete;
import org.evolizer.changedistiller.model.entities.Insert;
import org.evolizer.changedistiller.model.entities.Move;
import org.evolizer.changedistiller.model.entities.Update;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

public class ChangeClassifierTest extends AbstractDistillerTest {

    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
        TEST_FOLDER = "./test_data/classifier/";
        UNIQUE_NAME = "test.Test";
        MODIFIERS = ChangeModifier.PUBLIC | ChangeModifier.FINAL;
        AbstractDistillerTest.setUpBeforeClass();
    }

    @Test
    public void classifyOperationsTest() {
        int[] result = new int[ChangeType.values().length];
        List<SourceCodeChange> changes = fDJob.getSourceCodeChanges();
        for (SourceCodeChange scc : changes) {
            result[scc.getChangeType().ordinal()]++;
            switch (scc.getChangeType()) {
                case ADDING_ATTRIBUTE_MODIFIABILITY: // 1
                    Assert.assertEquals("test.Test.fAddingAttributeModifiability : String", scc.getRootEntity()
                            .getUniqueName());
                    break;
                case ADDING_CLASS_DERIVABILITY: // 2
                    Assert.assertEquals("test.Test.AddingClassDerivability", scc.getRootEntity().getUniqueName());
                    break;
                case ADDING_METHOD_OVERRIDABILITY: // 3
                    Assert.assertEquals("test.Test.addingMethodOverridability()", scc.getRootEntity().getUniqueName());
                    break;
                case ADDITIONAL_CLASS: // 4
                    Assert.assertEquals("test.Test", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("test.Test.AdditionalClass", scc.getChangedEntity().getUniqueName());
                    break;
                case ADDITIONAL_FUNCTIONALITY: // 5
                    Assert.assertEquals("test.Test", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("test.Test.additionalFunctionality()", scc.getChangedEntity().getUniqueName());
                    break;
                case ADDITIONAL_OBJECT_STATE: // 6
                    Assert.assertEquals("test.Test", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("test.Test.fAdditionalObjectState : int", scc.getChangedEntity()
                            .getUniqueName());
                    break;
                case ALTERNATIVE_PART_DELETE: // 7
                    Assert.assertEquals("test.Test.methodToCheckAlternativePartDelete()", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("alternativePartDelete == elseDelete", scc.getChangedEntity().getUniqueName());
                    break;
                case ALTERNATIVE_PART_INSERT: // 8
                    Assert.assertEquals("test.Test.methodToCheckAlternativePartInsert()", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("alternativePartInsert == elseInsert", scc.getChangedEntity().getUniqueName());
                    break;
                case ATTRIBUTE_RENAMING: // 9
                    Assert.assertEquals("test.Test.attributeRenaming : double", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("atributeRenamin", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("attributeRenaming", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case ATTRIBUTE_TYPE_CHANGE: // 10
                    Assert.assertEquals("test.Test.attributeTypeChange : Object", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("Integer", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("Object", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case COMMENT_DELETE: // 11
                    Assert.assertEquals("test.Test.methodToCheckCommentDelete()", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("// comment delete", scc.getChangedEntity().getUniqueName());
                    break;
                case COMMENT_INSERT: // 12
                    Assert.assertEquals("test.Test.methodToCheckCommentInsert()", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("// comment insert", scc.getChangedEntity().getUniqueName());
                    break;
                case COMMENT_MOVE: // 13
                    Assert.assertEquals("test.Test.methodToCheckCommentMove()", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("/* comment move */", scc.getChangedEntity().getUniqueName());
                    break;
                case COMMENT_UPDATE: // 14
                    Assert.assertEquals("test.Test.methodToCheckCommentUpdate()", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("/* comment that will be updated */", scc.getChangedEntity().getUniqueName());
                    Assert
                            .assertEquals("/* comment that was updated */", ((Update) scc).getNewEntity()
                                    .getUniqueName());
                    break;
                case CONDITION_EXPRESSION_CHANGE: // 15
                    Assert.assertEquals("test.Test.methodToCheckConditionExpressionChange()", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("conditioExpresionChang", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("conditionExpressionChange", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case DECREASING_ACCESSIBILITY_CHANGE: // 33
                    checkDecreasingAccessibilityChange(scc);
                    break;
                case DOC_DELETE: // 36
                    checkDocDelete(scc);
                    break;
                case DOC_INSERT: // 39
                    checkDocInsert(scc);
                    break;
                case DOC_UPDATE: // 42
                    checkDocUpdate(scc);
                    break;
                case METHOD_RENAMING: // 43
                    Assert.assertEquals("test.Test.methodRenaming()", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("metodRenamin", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("methodRenaming", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case PARAMETER_DELETE: // 44
                    Assert
                            .assertEquals("test.Test.methodToCheckParameterDelete()", scc.getRootEntity()
                                    .getUniqueName());
                    Assert.assertEquals("aInt", scc.getChangedEntity().getUniqueName());
                    break;
                case PARAMETER_INSERT: // 45
                    Assert.assertEquals("test.Test.methodToCheckParameterInsert(int)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("aInt", scc.getChangedEntity().getUniqueName());
                    break;
                case PARAMETER_ORDERING_CHANGE: // 46
                    Assert.assertEquals("test.Test.methodToCheckParameterOrderingChange(int, float, int)", scc
                            .getRootEntity().getUniqueName());
                    Assert.assertEquals("aMovedParam", scc.getChangedEntity().getUniqueName());
                    break;
                case PARAMETER_RENAMING: // 47
                    Assert.assertEquals("test.Test.methodToCheckParameterRenaming(Object)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("aRenamePara", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("aRenamedParam", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case PARAMETER_TYPE_CHANGE: // 48
                    Assert.assertEquals("test.Test.methodToCheckParameterTypeChange(Object, String)", scc
                            .getRootEntity().getUniqueName());
                    Assert.assertEquals("typeChange: int", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("typeChange: String", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case PARENT_CLASS_CHANGE: // 49
                    Assert.assertEquals("test.Test.ClassToCheckParentClassChange", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("Object", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("String", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case PARENT_CLASS_DELETE: // 50
                    Assert.assertEquals("test.Test.ClassToCheckParentClassDelete", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("ParentToDelete", scc.getChangedEntity().getUniqueName());
                    break;
                case PARENT_CLASS_INSERT: // 51
                    Assert.assertEquals("test.Test.ClassToCheckParentClassInsert", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("ParentToInsert", scc.getChangedEntity().getUniqueName());
                    break;
                case PARENT_INTERFACE_CHANGE: // 52
                    Assert.assertEquals("test.Test.ClassToCheckParentInterfaceChange", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("IIntefaceChang", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("IInterfaceChange", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case PARENT_INTERFACE_DELETE: // 53
                    Assert.assertEquals("test.Test.ClassToCheckParentInterfaceDelete", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("ParentInterfaceToDelete", scc.getChangedEntity().getUniqueName());
                    break;
                case PARENT_INTERFACE_INSERT: // 54
                    Assert.assertEquals("test.Test.ClassToCheckParentInterfaceInsert", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("ParentInterfaceToInsert", scc.getChangedEntity().getUniqueName());
                    break;
                case REMOVED_CLASS: // 55
                    Assert.assertEquals("test.Test", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("test.Test.RemovedClass", scc.getChangedEntity().getUniqueName());
                    break;
                case REMOVED_FUNCTIONALITY: // 56
                    Assert.assertEquals("test.Test", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckRemovedFunctionality()", scc.getChangedEntity()
                            .getUniqueName());
                    break;
                case REMOVED_OBJECT_STATE: // 57
                    Assert.assertEquals("test.Test", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("test.Test.fAttributeToCheckRemovedObjectState : List", scc.getChangedEntity()
                            .getUniqueName());
                    break;
                case REMOVING_ATTRIBUTE_MODIFIABILITY: // 58
                    Assert.assertEquals("test.Test.fRemovingAttributeModifiability : String", scc.getRootEntity()
                            .getUniqueName());
                    break;
                case REMOVING_CLASS_DERIVABILITY: // 59
                    Assert.assertEquals("test.Test.RemovingClassDerivability", scc.getRootEntity().getUniqueName());
                    break;
                case REMOVING_METHOD_OVERRIDABILITY: // 60
                    Assert
                            .assertEquals("test.Test.removingMethodOverridability()", scc.getRootEntity()
                                    .getUniqueName());
                    break;
                case RETURN_TYPE_CHANGE: // 61
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeChange()", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeChange(): Object", scc.getChangedEntity()
                            .getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeChange(): String", ((Update) scc)
                            .getNewEntity().getUniqueName());
                    break;
                case RETURN_TYPE_DELETE: // 62
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeDelete()", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeDelete(): int", scc.getChangedEntity()
                            .getUniqueName());
                    break;
                case RETURN_TYPE_INSERT: // 63
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeInsert()", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckReturnTypeInsert(): int", scc.getChangedEntity()
                            .getUniqueName());
                    break;
                case STATEMENT_DELETE: // 64
                    Assert.assertEquals("test.Test.methodToCheckStatementDelete(int)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("statement.delete();", scc.getChangedEntity().getUniqueName());
                    break;
                case STATEMENT_INSERT: // 65
                    Assert.assertEquals("test.Test.methodToCheckStatementInsert(int)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("statement.insert();", scc.getChangedEntity().getUniqueName());
                    break;
                case STATEMENT_ORDERING_CHANGE: // 66
                    Assert.assertEquals("test.Test.methodToCheckStatementOrderingChange(int)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckStatementOrderingChange(int)", ((Move) scc)
                            .getParentEntity().getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckStatementOrderingChange(int)", ((Move) scc)
                            .getNewParentEntity().getUniqueName());
                    Assert.assertEquals("statement.ordering();", scc.getChangedEntity().getUniqueName());
                    break;
                case STATEMENT_PARENT_CHANGE: // 67
                    Assert.assertEquals("test.Test.methodToCheckStatementParentChange(int)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("test.Test.methodToCheckStatementParentChange(int)", ((Move) scc)
                            .getParentEntity().getUniqueName());
                    Assert.assertEquals("daNewParent == true", ((Move) scc).getNewParentEntity().getUniqueName());
                    Assert.assertEquals("statement.parent();", scc.getChangedEntity().getUniqueName());
                    break;
                case STATEMENT_UPDATE: // 68
                    Assert.assertEquals("test.Test.methodToCheckStatementUpdate(int)", scc.getRootEntity()
                            .getUniqueName());
                    Assert.assertEquals("statment.updae();", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("statement.update();", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case INCREASING_ACCESSIBILITY_CHANGE: // 86
                    checkIncreasingAccessibilityChange(scc);
                    break;
                case CLASS_RENAMING: // 87
                    Assert.assertEquals("test.Test.ClassRenaming", scc.getRootEntity().getUniqueName());
                    Assert.assertEquals("ClssRenamin", scc.getChangedEntity().getUniqueName());
                    Assert.assertEquals("ClassRenaming", ((Update) scc).getNewEntity().getUniqueName());
                    break;
                case UNCLASSIFIED_CHANGE:
                    Assert.fail("Unclassified changes are not allowed: " + scc.toString());
            }
        }
        Assert.assertEquals(1, result[ChangeType.ADDING_ATTRIBUTE_MODIFIABILITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ADDING_CLASS_DERIVABILITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ADDING_METHOD_OVERRIDABILITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ADDITIONAL_CLASS.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ADDITIONAL_FUNCTIONALITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ADDITIONAL_OBJECT_STATE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ALTERNATIVE_PART_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ALTERNATIVE_PART_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ATTRIBUTE_RENAMING.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.ATTRIBUTE_TYPE_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.COMMENT_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.COMMENT_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.COMMENT_MOVE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.COMMENT_UPDATE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.CONDITION_EXPRESSION_CHANGE.ordinal()]);
        Assert.assertEquals(18, result[ChangeType.DECREASING_ACCESSIBILITY_CHANGE.ordinal()]);
        Assert.assertEquals(3, result[ChangeType.DOC_DELETE.ordinal()]);
        Assert.assertEquals(3, result[ChangeType.DOC_INSERT.ordinal()]);
        Assert.assertEquals(3, result[ChangeType.DOC_UPDATE.ordinal()]);
        Assert.assertEquals(18, result[ChangeType.INCREASING_ACCESSIBILITY_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.METHOD_RENAMING.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARAMETER_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARAMETER_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARAMETER_ORDERING_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARAMETER_RENAMING.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARAMETER_TYPE_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARENT_CLASS_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARENT_CLASS_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARENT_CLASS_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARENT_INTERFACE_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARENT_INTERFACE_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.PARENT_INTERFACE_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.REMOVED_CLASS.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.REMOVED_FUNCTIONALITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.REMOVED_OBJECT_STATE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.REMOVING_ATTRIBUTE_MODIFIABILITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.REMOVING_CLASS_DERIVABILITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.REMOVING_METHOD_OVERRIDABILITY.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.RETURN_TYPE_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.RETURN_TYPE_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.RETURN_TYPE_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.STATEMENT_DELETE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.STATEMENT_INSERT.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.STATEMENT_ORDERING_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.STATEMENT_PARENT_CHANGE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.STATEMENT_UPDATE.ordinal()]);
        Assert.assertEquals(1, result[ChangeType.CLASS_RENAMING.ordinal()]);
        Assert.assertEquals(87, sum(result));
    }

    private void checkDocDelete(SourceCodeChange scc) {
        switch (scc.getRootEntity().getType()) {
            case CLASS:
                Assert.assertEquals("test.Test.ClassToCheckDocDelete", scc.getRootEntity().getUniqueName());
                Assert.assertEquals("/**\n     * Doc to delete\n     */", scc.getChangedEntity().getUniqueName());
                break;
            case METHOD:
                Assert.assertEquals("test.Test.methodToCheckDocDelete()", scc.getRootEntity().getUniqueName());
                Assert.assertEquals("/**\n     * Doc to delete\n     */", scc.getChangedEntity().getUniqueName());
                break;
            case ATTRIBUTE:
                Assert.assertEquals("test.Test.fAttributeToCheckDocDelete : Object", scc.getRootEntity()
                        .getUniqueName());
                Assert.assertEquals("/**\n     * Doc to delete\n     */", scc.getChangedEntity().getUniqueName());
                break;
            default:
                Assert.fail("Given entity type fits not into doc delete: " + scc.getRootEntity().getType());
        }
    }

    private void checkDocInsert(SourceCodeChange scc) {
        switch (scc.getRootEntity().getType()) {
            case CLASS:
                Assert.assertEquals("test.Test.ClassToCheckDocInsert", scc.getRootEntity().getUniqueName());
                Assert.assertEquals("/**\n     * Doc to insert\n     */", scc.getChangedEntity().getUniqueName());
                break;
            case METHOD:
                Assert.assertEquals("test.Test.methodToCheckDocInsert()", scc.getRootEntity().getUniqueName());
                Assert.assertEquals("/**\n     * Doc to insert\n     */", scc.getChangedEntity().getUniqueName());
                break;
            case ATTRIBUTE:
                Assert.assertEquals("test.Test.fAttributeToCheckDocInsert : Object", scc.getRootEntity()
                        .getUniqueName());
                Assert.assertEquals("/**\n     * Doc to insert\n     */", scc.getChangedEntity().getUniqueName());
                break;
            default:
                Assert.fail("Given entity type fits not into doc delete: " + scc.getRootEntity().getType());
        }
    }

    private void checkDocUpdate(SourceCodeChange scc) {
        switch (scc.getRootEntity().getType()) {
            case CLASS:
                Assert.assertEquals("test.Test.ClassToCheckDocUpdate", scc.getRootEntity().getUniqueName());
                Assert.assertEquals("/**\n     * Doc that will be updated\n     */", scc.getChangedEntity()
                        .getUniqueName());
                Assert.assertEquals("/**\n     * Doc that was updated\n     */", ((Update) scc).getNewEntity()
                        .getUniqueName());
                break;
            case METHOD:
                Assert.assertEquals("test.Test.methodToCheckDocUpdate()", scc.getRootEntity().getUniqueName());
                Assert.assertEquals("/**\n     * Doc that will be updated\n     */", scc.getChangedEntity()
                        .getUniqueName());
                Assert.assertEquals("/**\n     * Doc that was updated\n     */", ((Update) scc).getNewEntity()
                        .getUniqueName());
                break;
            case ATTRIBUTE:
                Assert.assertEquals("test.Test.fAttributeToCheckDocUpdate : Object", scc.getRootEntity()
                        .getUniqueName());
                Assert.assertEquals("/**\n     * Doc that will be updated\n     */", scc.getChangedEntity()
                        .getUniqueName());
                Assert.assertEquals("/**\n     * Doc that was updated\n     */", ((Update) scc).getNewEntity()
                        .getUniqueName());
                break;
            default:
                Assert.fail("Given entity type fits not into doc update: " + scc.getRootEntity().getType());
        }
    }

    private int sum(int[] array) {
        int result = 0;
        for (int n : array) {
            result += n;
        }
        return result;
    }

    private void checkDecreasingAccessibilityChange(SourceCodeChange scc) {
        switch (scc.getRootEntity().getType()) {
            case CLASS:
                if (scc.getChangedEntity().getUniqueName().equals("public")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("protected")) {
                            checkDecreasingAccessibilityChangeFromPublicToProtected(
                                    "test.Test.DecreasingAccessibilityChangeFromPublicToProtected",
                                    upd);
                        } else if (newUniqueName.equals("private")) {
                            checkDecreasingAccessibilityChangeFromPublicToPrivate(
                                    "test.Test.DecreasingAccessibilityChangeFromPublicToPrivate",
                                    upd);
                        } else {
                            Assert.fail("Class decreasing accessibility change from public: invalid source code change"
                                    + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkDecreasingAccessibilityChangeFromPublicToPackage(
                                "test.Test.DecreasingAccessibilityChangeFromPublicToPackage",
                                scc);
                    } else {
                        Assert.fail("Class decreasing accessibility change from public: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("protected")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("private")) {
                            checkDecreasingAccessibilityChangeFromProtectedToPrivate(
                                    "test.Test.DecreasingAccessibilityChangeFromProtectedToPrivate",
                                    upd);
                        } else {
                            Assert
                                    .fail("Class decreasing accessibility change from protected: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkDecreasingAccessibilityChangeFromProtectedToPackage(
                                "test.Test.DecreasingAccessibilityChangeFromProtectedToPackage",
                                scc);
                    } else {
                        Assert.fail("Class decreasing accessibility change from protected: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("private")) {
                    if (scc instanceof Insert) {
                        checkDecreasingAccessibilityChangeFromPackageToPrivate(
                                "test.Test.DecreasingAccessibilityChangeFromPackageToPrivate",
                                scc);
                    } else {
                        Assert.fail("Class decreasing accessibility change from package: invalid source code change"
                                + scc.toString());
                    }
                }
                break;
            case METHOD:
                if (scc.getChangedEntity().getUniqueName().equals("public")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("protected")) {
                            checkDecreasingAccessibilityChangeFromPublicToProtected(
                                    "test.Test.decreasingAccessibilityChangeFromPublicToProtected()",
                                    upd);
                        } else if (newUniqueName.equals("private")) {
                            checkDecreasingAccessibilityChangeFromPublicToPrivate(
                                    "test.Test.decreasingAccessibilityChangeFromPublicToPrivate()",
                                    upd);
                        } else {
                            Assert
                                    .fail("Method decreasing accessibility change from public: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkDecreasingAccessibilityChangeFromPublicToPackage(
                                "test.Test.decreasingAccessibilityChangeFromPublicToPackage()",
                                scc);
                    } else {
                        Assert.fail("Method decreasing accessibility change from public: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("protected")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("private")) {
                            checkDecreasingAccessibilityChangeFromProtectedToPrivate(
                                    "test.Test.decreasingAccessibilityChangeFromProtectedToPrivate()",
                                    upd);
                        } else {
                            Assert
                                    .fail("Method decreasing accessibility change from protected: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkDecreasingAccessibilityChangeFromProtectedToPackage(
                                "test.Test.decreasingAccessibilityChangeFromProtectedToPackage()",
                                scc);
                    } else {
                        Assert.fail("Method decreasing accessibility change from protected: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("private")) {
                    if (scc instanceof Insert) {
                        checkDecreasingAccessibilityChangeFromPackageToPrivate(
                                "test.Test.decreasingAccessibilityChangeFromPackageToPrivate()",
                                scc);
                    } else {
                        Assert.fail("Method decreasing accessibility change from package: invalid source code change"
                                + scc.toString());
                    }
                }
                break;
            case ATTRIBUTE:
                if (scc.getChangedEntity().getUniqueName().equals("public")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("protected")) {
                            checkDecreasingAccessibilityChangeFromPublicToProtected(
                                    "test.Test.fDecreasingAccessibilityChangeFromPublicToProtected : Object",
                                    upd);
                        } else if (newUniqueName.equals("private")) {
                            checkDecreasingAccessibilityChangeFromPublicToPrivate(
                                    "test.Test.fDecreasingAccessibilityChangeFromPublicToPrivate : Object",
                                    upd);
                        } else {
                            Assert
                                    .fail("Attribute decreasing accessibility change from public: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkDecreasingAccessibilityChangeFromPublicToPackage(
                                "test.Test.fDecreasingAccessibilityChangeFromPublicToPackage : Object",
                                scc);
                    } else {
                        Assert.fail("Attribute decreasing accessibility change from public: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("protected")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("private")) {
                            checkDecreasingAccessibilityChangeFromProtectedToPrivate(
                                    "test.Test.fDecreasingAccessibilityChangeFromProtectedToPrivate : Object",
                                    upd);
                        } else {
                            Assert
                                    .fail("Attribute decreasing accessibility change from protected: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkDecreasingAccessibilityChangeFromProtectedToPackage(
                                "test.Test.fDecreasingAccessibilityChangeFromProtectedToPackage : Object",
                                scc);
                    } else {
                        Assert
                                .fail("Attribute decreasing accessibility change from protected: invalid source code change"
                                        + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("private")) {
                    if (scc instanceof Insert) {
                        checkDecreasingAccessibilityChangeFromPackageToPrivate(
                                "test.Test.fDecreasingAccessibilityChangeFromPackageToPrivate : Object",
                                scc);
                    } else {
                        Assert
                                .fail("Attribute decreasing accessibility change from package: invalid source code change"
                                        + scc.toString());
                    }
                }
                break;
            default:
                Assert.fail("Given entity type fits not into decreasing accessibility changes: "
                        + scc.getRootEntity().getType());
        }
    }

    private void checkIncreasingAccessibilityChange(SourceCodeChange scc) {
        switch (scc.getRootEntity().getType()) {
            case CLASS:
                if (scc.getChangedEntity().getUniqueName().equals("private")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("protected")) {
                            checkIncreasingAccessibilityChangeFromPrivateToProtected(
                                    "test.Test.IncreasingAccessibilityChangeFromPrivateToProtected",
                                    upd);
                        } else if (newUniqueName.equals("public")) {
                            checkIncreasingAccessibilityChangeFromPrivateToPublic(
                                    "test.Test.IncreasingAccessibilityChangeFromPrivateToPublic",
                                    upd);
                        } else {
                            Assert
                                    .fail("Class increasing accessibility change from private: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkIncreasingAccessibilityChangeFromPrivateToPackage(
                                "test.Test.IncreasingAccessibilityChangeFromPrivateToPackage",
                                scc);
                    } else {
                        Assert.fail("Class increasing accessibility change from private: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("protected")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        checkIncreasingAccessibilityChangeFromProtectedToPublic(
                                "test.Test.IncreasingAccessibilityChangeFromProtectedToPublic",
                                upd);
                    } else if (scc instanceof Insert) {
                        checkIncreasingAccessibilityChangeFromPackageToProtected(
                                "test.Test.IncreasingAccessibilityChangeFromPackageToProtected",
                                scc);
                    } else {
                        Assert.fail("Class increasing accessibility change from protected: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("public")) {
                    if (scc instanceof Insert) {
                        checkIncreasingAccessibilityChangeFromPackageToPublic(
                                "test.Test.IncreasingAccessibilityChangeFromPackageToPublic",
                                scc);
                    } else {
                        Assert.fail("Class increasing accessibility change from package: invalid source code change "
                                + scc.toString());
                    }
                }
                break;
            case METHOD:
                if (scc.getChangedEntity().getUniqueName().equals("private")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("protected")) {
                            checkIncreasingAccessibilityChangeFromPrivateToProtected(
                                    "test.Test.increasingAccessibilityChangeFromPrivateToProtected()",
                                    upd);
                        } else if (newUniqueName.equals("public")) {
                            checkIncreasingAccessibilityChangeFromPrivateToPublic(
                                    "test.Test.increasingAccessibilityChangeFromPrivateToPublic()",
                                    upd);
                        } else {
                            Assert
                                    .fail("Method increasing accessibility change from private: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkIncreasingAccessibilityChangeFromPrivateToPackage(
                                "test.Test.increasingAccessibilityChangeFromPrivateToPackage()",
                                scc);
                    } else {
                        Assert.fail("Method increasing accessibility change from private: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("protected")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        checkIncreasingAccessibilityChangeFromProtectedToPublic(
                                "test.Test.increasingAccessibilityChangeFromProtectedToPublic()",
                                upd);
                    } else if (scc instanceof Insert) {
                        checkIncreasingAccessibilityChangeFromPackageToProtected(
                                "test.Test.increasingAccessibilityChangeFromPackageToProtected()",
                                scc);
                    } else {
                        Assert.fail("Method increasing accessibility change from protected: invalid source code change"
                                + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("public")) {
                    if (scc instanceof Insert) {
                        checkIncreasingAccessibilityChangeFromPackageToPublic(
                                "test.Test.increasingAccessibilityChangeFromPackageToPublic()",
                                scc);
                    } else {
                        Assert.fail("Method increasing accessibility change from package: invalid source code change "
                                + scc.toString());
                    }
                }
                break;
            case ATTRIBUTE:
                if (scc.getChangedEntity().getUniqueName().equals("private")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        String newUniqueName = upd.getNewEntity().getUniqueName();
                        if (newUniqueName.equals("protected")) {
                            checkIncreasingAccessibilityChangeFromPrivateToProtected(
                                    "test.Test.fIncreasingAccessibilityChangeFromPrivateToProtected : Object",
                                    upd);
                        } else if (newUniqueName.equals("public")) {
                            checkIncreasingAccessibilityChangeFromPrivateToPublic(
                                    "test.Test.fIncreasingAccessibilityChangeFromPrivateToPublic : Object",
                                    upd);
                        } else {
                            Assert
                                    .fail("Attribute increasing accessibility change from private: invalid source code change"
                                            + scc.toString());
                        }
                    } else if (scc instanceof Delete) {
                        checkIncreasingAccessibilityChangeFromPrivateToPackage(
                                "test.Test.fIncreasingAccessibilityChangeFromPrivateToPackage : Object",
                                scc);
                    } else {
                        Assert
                                .fail("Attribute increasing accessibility change from private: invalid source code change"
                                        + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("protected")) {
                    if (scc instanceof Update) {
                        Update upd = (Update) scc;
                        checkIncreasingAccessibilityChangeFromProtectedToPublic(
                                "test.Test.fIncreasingAccessibilityChangeFromProtectedToPublic : Object",
                                upd);
                    } else if (scc instanceof Insert) {
                        checkIncreasingAccessibilityChangeFromPackageToProtected(
                                "test.Test.fIncreasingAccessibilityChangeFromPackageToProtected : Object",
                                scc);
                    } else {
                        Assert
                                .fail("Attribute increasing accessibility change from protected: invalid source code change"
                                        + scc.toString());
                    }
                } else if (scc.getChangedEntity().getUniqueName().equals("public")) {
                    if (scc instanceof Insert) {
                        checkIncreasingAccessibilityChangeFromPackageToPublic(
                                "test.Test.fIncreasingAccessibilityChangeFromPackageToPublic : Object",
                                scc);
                    } else {
                        Assert
                                .fail("Attribute increasing accessibility change from package: invalid source code change "
                                        + scc.toString());
                    }
                }
                break;
            default:
                Assert.fail("Given entity type fits not into increasing accessibility changes: "
                        + scc.getRootEntity().getType());
        }
    }

    private void checkDecreasingAccessibilityChangeFromPublicToProtected(String rootEntityName, Update update) {
        Assert.assertEquals(rootEntityName, update.getRootEntity().getUniqueName());
        Assert.assertEquals("public", update.getChangedEntity().getUniqueName());
        Assert.assertEquals("protected", update.getNewEntity().getUniqueName());
    }

    private void checkDecreasingAccessibilityChangeFromPublicToPrivate(String rootEntityName, Update update) {
        Assert.assertEquals(rootEntityName, update.getRootEntity().getUniqueName());
        Assert.assertEquals("public", update.getChangedEntity().getUniqueName());
        Assert.assertEquals("private", update.getNewEntity().getUniqueName());
    }

    private void checkDecreasingAccessibilityChangeFromPublicToPackage(
            String rootEntityName,
            SourceCodeChange scc) {
        Assert.assertEquals(rootEntityName, scc.getRootEntity().getUniqueName());
        Assert.assertEquals("public", scc.getChangedEntity().getUniqueName());
    }

    private void checkDecreasingAccessibilityChangeFromProtectedToPackage(
            String rootEntityName,
            SourceCodeChange scc) {
        Assert.assertEquals(rootEntityName, scc.getRootEntity().getUniqueName());
        Assert.assertEquals("protected", scc.getChangedEntity().getUniqueName());
    }

    private void checkDecreasingAccessibilityChangeFromProtectedToPrivate(String rootEntityName, Update update) {
        Assert.assertEquals(rootEntityName, update.getRootEntity().getUniqueName());
        Assert.assertEquals("protected", update.getChangedEntity().getUniqueName());
        Assert.assertEquals("private", update.getNewEntity().getUniqueName());
    }

    private void checkDecreasingAccessibilityChangeFromPackageToPrivate(
            String rootEntityName,
            SourceCodeChange scc) {
        Assert.assertEquals(rootEntityName, scc.getRootEntity().getUniqueName());
        Assert.assertEquals("private", scc.getChangedEntity().getUniqueName());
    }

    private void checkIncreasingAccessibilityChangeFromProtectedToPublic(String rootEntityName, Update update) {
        Assert.assertEquals(rootEntityName, update.getRootEntity().getUniqueName());
        Assert.assertEquals("protected", update.getChangedEntity().getUniqueName());
        Assert.assertEquals("public", update.getNewEntity().getUniqueName());
    }

    private void checkIncreasingAccessibilityChangeFromPrivateToPublic(String rootEntityName, Update update) {
        Assert.assertEquals(rootEntityName, update.getRootEntity().getUniqueName());
        Assert.assertEquals("private", update.getChangedEntity().getUniqueName());
        Assert.assertEquals("public", update.getNewEntity().getUniqueName());
    }

    private void checkIncreasingAccessibilityChangeFromPackageToPublic(
            String rootEntityName,
            SourceCodeChange scc) {
        Assert.assertEquals(rootEntityName, scc.getRootEntity().getUniqueName());
        Assert.assertEquals("public", scc.getChangedEntity().getUniqueName());
    }

    private void checkIncreasingAccessibilityChangeFromPackageToProtected(
            String rootEntityName,
            SourceCodeChange scc) {
        Assert.assertEquals(rootEntityName, scc.getRootEntity().getUniqueName());
        Assert.assertEquals("protected", scc.getChangedEntity().getUniqueName());
    }

    private void checkIncreasingAccessibilityChangeFromPrivateToProtected(String rootEntityName, Update update) {
        Assert.assertEquals(rootEntityName, update.getRootEntity().getUniqueName());
        Assert.assertEquals("private", update.getChangedEntity().getUniqueName());
        Assert.assertEquals("protected", update.getNewEntity().getUniqueName());
    }

    private void checkIncreasingAccessibilityChangeFromPrivateToPackage(
            String rootEntityName,
            SourceCodeChange scc) {
        Assert.assertEquals(rootEntityName, scc.getRootEntity().getUniqueName());
        Assert.assertEquals("private", scc.getChangedEntity().getUniqueName());
    }
}
