package org.evolizer.changedistiller.test.roundtrip;

import static org.junit.Assert.assertNotNull;

import java.io.InputStream;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.internal.ui.util.CoreUtility;
import org.evolizer.changedistiller.distilling.Distiller;
import org.evolizer.changedistiller.model.entities.AbstractHistory;
import org.evolizer.changedistiller.model.entities.AttributeHistory;
import org.evolizer.changedistiller.model.entities.ClassHistory;
import org.evolizer.changedistiller.model.entities.Delete;
import org.evolizer.changedistiller.model.entities.Insert;
import org.evolizer.changedistiller.model.entities.Move;
import org.evolizer.changedistiller.model.entities.SourceCodeChange;
import org.evolizer.changedistiller.model.entities.StructureEntityVersion;
import org.evolizer.changedistiller.model.entities.Update;
import org.evolizer.changedistiller.test.Activator;
import org.evolizer.core.exceptions.EvolizerException;
import org.evolizer.core.hibernate.model.api.IEvolizerModelEntity;
import org.evolizer.core.hibernate.session.EvolizerSessionHandler;
import org.evolizer.core.hibernate.session.api.IEvolizerSession;
import org.evolizer.core.util.projecthandling.JavaProjectHelper;
import org.evolizer.model.resources.entities.humans.Person;
import org.evolizer.versioncontrol.cvs.model.entities.ModificationReport;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

@SuppressWarnings("restriction")
public class ChangeDistillerRoundTripTest {

    private static final String DB_CONFIG_FILE = "./config/db.properties";
    private static String fDBUrl;
    private static String fDBHost;
    private static String fDBName;
    private static String fDBDialect;
    private static String fDBDriverName;
    private static String fDBUser;
    private static String fDBPasswd;
    private static EvolizerSessionHandler fSessionHandler = null;
    private static IEvolizerSession fEvoSession = null;
    private static JavaProjectHelper fHelper;
    private static IJavaProject fJProject;
    private static IFolder fFolder;

    private Revision fRevision2;
    private Revision fRevision1;
    private static String TEST_FOLDER = "./test_data/roundtrip/";

    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
        InputStream propertiesInputStream = Activator.openBundledFile(DB_CONFIG_FILE);

        if (propertiesInputStream != null) {
            Properties props = new Properties();
            props.load(propertiesInputStream);
            propertiesInputStream.close();

            fDBHost = props.getProperty("dbHost");
            fDBName = props.getProperty("dbName");
            fDBUrl = fDBHost + "/" + fDBName;
            fDBDialect = props.getProperty("dbDialect");
            fDBDriverName = props.getProperty("dbDriverName");
            fDBUser = props.getProperty("dbUser");
            fDBPasswd = props.getProperty("dbPasswd");
        }

        // creating an empty dummy Java project
        fHelper = new JavaProjectHelper();
        fHelper.createProject("TestProject", "bin", new NullProgressMonitor());

        // creating a folder for the files to compare
        fHelper.addStandartSourceFolder(new NullProgressMonitor());
        fHelper.addPackage("", new NullProgressMonitor());
        fJProject = fHelper.getJavaProject();
        assertNotNull(fJProject);
        assertNotNull(fJProject);
        fFolder = fJProject.getProject().getFolder("folder");
        CoreUtility.createFolder(fFolder, true, true, null);
    }

    @AfterClass
    public static void tearDownAfterClass() throws Exception {
        fHelper.deleteWorkspace(true, true, new NullProgressMonitor());
    }

    @Before
    public void setUp() {
        try {
            fSessionHandler = EvolizerSessionHandler.getHandler();
            fSessionHandler.initSessionFactory(fDBUrl, fDBUser, fDBPasswd);
            fSessionHandler.createSchema(fDBUrl, fDBDialect, fDBDriverName, fDBUser, fDBPasswd);
            fEvoSession = fSessionHandler.getCurrentSession(fDBUrl);

            fRevision1 = new Revision("1.2");
            fRevision1.setReport(new ModificationReport(new Date()));
            fRevision1.setAuthor(new Person());
            saveToDB(fRevision1);

            fRevision2 = new Revision("1.3");
            fRevision2.setReport(new ModificationReport(new Date()));
            fRevision2.setAuthor(new Person());
            saveToDB(fRevision2);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @After
    public void tearDown() throws Exception {
        if (fEvoSession.isOpen()) {
            fEvoSession.flush();
            fEvoSession.close();
        }
        fSessionHandler.dropSchema(fDBUrl, fDBDialect, fDBDriverName, fDBUser, fDBPasswd);
    }

    @Test
    public void distillerRoundTripTest1() throws Exception {
        IFile v1 = fFolder.getFile("Test1.java");
        v1.create(Activator.openBundledFile(TEST_FOLDER + "Test1.java"), true, null);
        IFile v2 = fFolder.getFile("Test2.java");
        v2.create(Activator.openBundledFile(TEST_FOLDER + "Test2.java"), true, null);
        IFile v3 = fFolder.getFile("Test3.java");
        v3.create(Activator.openBundledFile(TEST_FOLDER + "Test3.java"), true, null);

        ClassHistory classHistory = null;

        // 1.1 -> 1.2
        Distiller distiller = new Distiller();
        distiller.performDistilling(v1, v2);
        distiller.getClassHistory().updateLatestVersionWithRevision(fRevision1);

        // 1.2 -> 1.3
        classHistory = distiller.getClassHistory();
        distiller = new Distiller();
        distiller.setClassHistory(classHistory);
        distiller.performDistilling(v2, v3);
        distiller.getClassHistory().updateLatestVersionWithRevision(fRevision2);

        if (classHistory != null) {
            if (classHistory.hasChanges()) {
                checkClassHistories(classHistory, saveAndUniqueReload(
                        classHistory,
                        "from ClassHistory where uniqueName='test.Test'",
                        ClassHistory.class));
            }
        }
    }

    @Test
    public void distillerRoundTripTest2() throws Exception {
        IFile v1 = fFolder.getFile("TestLeft.java");
        v1.create(Activator.openBundledFile(TEST_FOLDER + "TestLeft.java"), true, null);
        IFile v2 = fFolder.getFile("TestRight.java");
        v2.create(Activator.openBundledFile(TEST_FOLDER + "TestRight.java"), true, null);

        ClassHistory classHistory = null;

        // 1.1 -> 1.2
        Distiller distiller = new Distiller();
        distiller.performDistilling(v1, v2);
        distiller.getClassHistory().updateLatestVersionWithRevision(fRevision1);
        classHistory = distiller.getClassHistory();

        if (classHistory != null) {
            if (classHistory.hasChanges()) {
                checkClassHistories(classHistory, saveAndUniqueReload(
                        classHistory,
                        "from ClassHistory where uniqueName='test.Test'",
                        ClassHistory.class));
            }
        }
    }

    private void checkClassHistories(ClassHistory expected, ClassHistory result) {
        List<IEvolizerModelEntity> expectedEntities = split(expected);
        List<IEvolizerModelEntity> resultEntities = split(result);

        for (IEvolizerModelEntity resultEntity : resultEntities) {
            Assert.assertTrue(expectedEntities.remove(resultEntity));
        }
        Assert.assertTrue(expectedEntities.isEmpty());
    }

    private List<IEvolizerModelEntity> split(ClassHistory history) {
        List<IEvolizerModelEntity> result = new LinkedList<IEvolizerModelEntity>();
        attachFromAbstractHistory(history, result);
        attachFromHistories(history.getAttributeHistories(), result);
        return result;
    }

    private void attachFromHistories(Map<String, AttributeHistory> histories, List<IEvolizerModelEntity> list) {
        for (AbstractHistory history : histories.values()) {
            attachFromAbstractHistory(history, list);
        }
    }

    private void attachFromAbstractHistory(AbstractHistory history, List<IEvolizerModelEntity> list) {
        for (StructureEntityVersion sev : history.getVersions()) {
            list.add(sev);
            attachFromStructureEntityVersion(sev, list);
        }
    }

    private void attachFromStructureEntityVersion(StructureEntityVersion sev, List<IEvolizerModelEntity> list) {
        for (SourceCodeChange scc : sev.getSourceCodeChanges()) {
            list.add(scc);
            list.add(scc.getChangedEntity());
            attachFromSourceCodeChange(scc, list);
        }
    }

    private void attachFromSourceCodeChange(SourceCodeChange scc, List<IEvolizerModelEntity> list) {
        if (scc instanceof Insert) {
            Insert insert = (Insert) scc;
            list.add(insert.getParentEntity());
        } else if (scc instanceof Delete) {
            Delete delete = (Delete) scc;
            list.add(delete.getParentEntity());
        } else if (scc instanceof Move) {
            Move move = (Move) scc;
            list.add(move.getNewEntity());
            list.add(move.getParentEntity());
            list.add(move.getNewParentEntity());
        } else {
            Update update = (Update) scc;
            list.add(update.getParentEntity());
            list.add(update.getNewEntity());
        }
    }

    private void saveToDB(Object... objects) throws EvolizerException {
        fEvoSession.startTransaction();
        try {
            for (Object o : objects) {
                fEvoSession.saveObject(o);
            }
        } catch (Throwable t) {
            fEvoSession.rollbackTransaction();
            t.printStackTrace();
        } finally {
            fEvoSession.endTransaction();
        }
        fEvoSession.close();
        fEvoSession = fSessionHandler.getCurrentSession(fDBUrl);
    }

    private <T> T saveAndUniqueReload(Object object, String query, Class<T> resultType) throws EvolizerException {
        saveToDB(object);
        return reloadUniqueFromDB(query, resultType);
    }

    private <T> T reloadUniqueFromDB(String query, Class<T> resultType) {
        return fEvoSession.uniqueResult(query, resultType);
    }

}
