package org.evolizer.changedistiller.test.roundtrip;

import java.io.FileInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Properties;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jdt.core.IJavaProject;
import org.evolizer.changedistiller.job.ChangeDistiller;
import org.evolizer.changedistiller.test.Activator;
import org.evolizer.core.hibernate.session.EvolizerSessionHandler;
import org.evolizer.core.hibernate.session.api.IEvolizerSession;
import org.evolizer.core.preferences.EvolizerPreferences;
import org.evolizer.core.util.projecthandling.JavaProjectHelper;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class AutomatedChangeDistillerRoundTripTest {

    private static final String DB_CONFIG_FILE = "./config/db.properties";
    private static final String MYSQL = "/usr/local/mysql/bin/mysql";
    private static String fDBHost;
    private static String fDBName;
    private static String fDBUser;
    private static String fDBPasswd;
    private static String fDBDump;
    private IEvolizerSession fEvoSession;
    private static IJavaProject fJProject;
    private static JavaProjectHelper fHelper;

    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
        InputStream propertiesInputStream = Activator.openBundledFile(DB_CONFIG_FILE);

        if (propertiesInputStream != null) {
            Properties props = new Properties();
            props.load(propertiesInputStream);
            propertiesInputStream.close();

            fDBHost = props.getProperty("dbHost");
            fDBName = props.getProperty("dbName");
            fDBUser = props.getProperty("dbUser");
            fDBPasswd = props.getProperty("dbPasswd");
            fDBDump = props.getProperty("dbDump");
        }

        fHelper = new JavaProjectHelper();
        fHelper.createProject("changedistiller_test", "bin", new NullProgressMonitor());
        fHelper.addStandartSourceFolder(new NullProgressMonitor());
        fHelper.addPackage("test", new NullProgressMonitor());
        fHelper.addSourceFile(
                "test",
                "Test.java",
                new FileInputStream("test_data/roundtrip/test/Test.java"),
                new NullProgressMonitor());
        fHelper.addSourceFile("test", "TestClassifier.java", new FileInputStream(
                "test_data/roundtrip/test/TestClassifier.java"), new NullProgressMonitor());
        fJProject = fHelper.getJavaProject();
        fJProject.open(new NullProgressMonitor());

        IProject project = fJProject.getProject();
        project.setPersistentProperty(EvolizerPreferences.DB_HOST, fDBHost);
        project.setPersistentProperty(EvolizerPreferences.DB_NAME, fDBName);
        project.setPersistentProperty(EvolizerPreferences.DB_USER, fDBUser);
        project.setPersistentProperty(EvolizerPreferences.DB_PASSWORD, fDBPasswd);
    }

    @AfterClass
    public static void tearDownAfterClass() throws Exception {}

    @Before
    public void setUp() throws Exception {
        String exec = MYSQL + " -u" + fDBUser + " -p" + fDBPasswd + " " + fDBName;
        Process process = Runtime.getRuntime().exec(exec);

        OutputStream stdout = process.getOutputStream();
        FileInputStream fin = new FileInputStream(fDBDump);
        int len = 0;
        byte[] b = new byte[8192];

        while ((len = fin.read(b, 0, 8192)) > 0) {
            stdout.write(b, 0, len);
            stdout.flush();
        }
        fin.close();
        stdout.close();
        process.waitFor();
    }

    @After
    public void tearDown() throws Exception {
        if (fEvoSession.isOpen()) {
            fEvoSession.flush();
            fEvoSession.close();
        }
    }

    @Test
    public void distillProjectTest() throws Exception {
        ChangeDistiller distiller = new ChangeDistiller();
        distiller.process(fJProject);
        Job[] job = Job.getJobManager().find("ChangeDistiller");
        job[0].join();
        fEvoSession = EvolizerSessionHandler.getHandler().getCurrentSession(fJProject.getProject());
        Assert.assertEquals(new Long(87), fEvoSession.uniqueResult("select count(*) from AbstractHistory", Long.class));
        Assert.assertEquals(new Long(92), fEvoSession.uniqueResult(
                "select count(*) from StructureEntityVersion",
                Long.class));
        Assert.assertEquals(new Long(275), fEvoSession
                .uniqueResult("select count(*) from SourceCodeEntity", Long.class));
        Assert.assertEquals(new Long(114), fEvoSession
                .uniqueResult("select count(*) from SourceCodeChange", Long.class));
    }

    @Test
    public void distillPackageFragmentTest() throws Exception {
        ChangeDistiller distiller = new ChangeDistiller();
        distiller.process(fJProject.findElement(new Path("test")));
        Job[] job = Job.getJobManager().find("ChangeDistiller");
        job[0].join();
        fEvoSession = EvolizerSessionHandler.getHandler().getCurrentSession(fJProject.getProject());
        Assert.assertEquals(new Long(87), fEvoSession.uniqueResult("select count(*) from AbstractHistory", Long.class));
        Assert.assertEquals(new Long(92), fEvoSession.uniqueResult(
                "select count(*) from StructureEntityVersion",
                Long.class));
        Assert.assertEquals(new Long(275), fEvoSession
                .uniqueResult("select count(*) from SourceCodeEntity", Long.class));
        Assert.assertEquals(new Long(114), fEvoSession
                .uniqueResult("select count(*) from SourceCodeChange", Long.class));
    }

    @Test
    public void distillSource1Test() throws Exception {
        ChangeDistiller distiller = new ChangeDistiller();
        distiller.process(fJProject.findElement(new Path("test/Test.java")));
        Job[] job = Job.getJobManager().find("ChangeDistiller");
        job[0].join();
        fEvoSession = EvolizerSessionHandler.getHandler().getCurrentSession(fJProject.getProject());
        Assert.assertEquals(new Long(6), fEvoSession.uniqueResult("select count(*) from AbstractHistory", Long.class));
        Assert.assertEquals(new Long(11), fEvoSession.uniqueResult(
                "select count(*) from StructureEntityVersion",
                Long.class));
        Assert
                .assertEquals(new Long(65), fEvoSession.uniqueResult(
                        "select count(*) from SourceCodeEntity",
                        Long.class));
        Assert
                .assertEquals(new Long(28), fEvoSession.uniqueResult(
                        "select count(*) from SourceCodeChange",
                        Long.class));
    }

    @Test
    public void distillSource2Test() throws Exception {
        ChangeDistiller distiller = new ChangeDistiller();
        distiller.process(fJProject.findElement(new Path("test/TestClassifier.java")));
        Job[] job = Job.getJobManager().find("ChangeDistiller");
        job[0].join();
        fEvoSession = EvolizerSessionHandler.getHandler().getCurrentSession(fJProject.getProject());
        Assert.assertEquals(new Long(81), fEvoSession.uniqueResult("select count(*) from AbstractHistory", Long.class));
        Assert.assertEquals(new Long(81), fEvoSession.uniqueResult(
                "select count(*) from StructureEntityVersion",
                Long.class));
        Assert.assertEquals(new Long(210), fEvoSession
                .uniqueResult("select count(*) from SourceCodeEntity", Long.class));
        Assert
                .assertEquals(new Long(86), fEvoSession.uniqueResult(
                        "select count(*) from SourceCodeChange",
                        Long.class));
    }

    @Test
    public void distillBothSourceTest() throws Exception {
        ChangeDistiller distiller = new ChangeDistiller();
        distiller.process(fJProject.findElement(new Path("test/TestClassifier.java")), fJProject.findElement(new Path(
                "test/Test.java")));
        Job[] job = Job.getJobManager().find("ChangeDistiller");
        job[0].join();
        fEvoSession = EvolizerSessionHandler.getHandler().getCurrentSession(fJProject.getProject());
        Assert.assertEquals(new Long(87), fEvoSession.uniqueResult("select count(*) from AbstractHistory", Long.class));
        Assert.assertEquals(new Long(92), fEvoSession.uniqueResult(
                "select count(*) from StructureEntityVersion",
                Long.class));
        Assert.assertEquals(new Long(275), fEvoSession
                .uniqueResult("select count(*) from SourceCodeEntity", Long.class));
        Assert.assertEquals(new Long(114), fEvoSession
                .uniqueResult("select count(*) from SourceCodeChange", Long.class));
    }
}
