/*
 * Copyright 2009 University of Zurich, Switzerland
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.evolizer.changedistiller.model.entities;

import java.util.LinkedList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Transient;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.core.exceptions.EvolizerRuntimeException;
import org.evolizer.core.hibernate.model.api.IEvolizerModelEntity;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;

/**
 * A structure entity version consists of all {@link SourceCodeChange}s applied in a version of an attribute, a class,
 * or a method.
 * <p>
 * For each new version of an attribute, a class, or a method, a new structure entity version is created.
 * 
 * @author fluri, zubi
 */
@Entity
public class StructureEntityVersion implements IEvolizerModelEntity {

    private static final String STRING = "@";

    private static final String NULL = "null";

    /**
     * Unique ID, used by Hibernate.
     */
    private Long fId = -1L;

    /**
     * Type code of structure entity.
     */
    private EntityType fType;

    /**
     * Structure entity's unique name.
     */
    private String fUniqueName;

    /**
     * Significance level of structure entity. Value is calculated not saved.
     */
    private Integer fSignificanceLevel = -1;

    /**
     * Modifiers of this entity.
     */
    private Integer fModifiers = 0;

    /**
     * Revision to which structure entity version belongs to. Link to <code>org.evolizer.model.versioning</code>.
     */
    private Revision fRevision;

    /**
     * {@link List} of {@link SourceCodeChange}s applied on structure entity.
     */
    private List<SourceCodeChange> fSourceCodeChanges;

    /**
     * Default constructor. Only used by Hibernate.
     */
    private StructureEntityVersion() {
        setSourceCodeChanges(new LinkedList<SourceCodeChange>());
    }

    /**
     * Constructor to initialize structure entity version with an entity and a version.
     * 
     * @param type
     *            of the structure entity version
     * @param uniqueName
     *            of the structure entity version
     * @param modifiers
     *            the modifiers
     * @param revision
     *            the revision
     */
    public StructureEntityVersion(EntityType type, String uniqueName, int modifiers, Revision revision) {
        this();
        setType(type);
        setUniqueName(uniqueName);
        setModifiers(modifiers);
        setRevision(revision);
    }

    /**
     * Instantiates a new structure entity version.
     * 
     * @param type
     *            the type
     * @param uniqueName
     *            the unique name
     * @param modifiers
     *            the modifiers
     */
    public StructureEntityVersion(EntityType type, String uniqueName, int modifiers) {
        this(type, uniqueName, modifiers, null);
    }

    /**
     * Unique ID, used by Hibernate.
     * 
     * @return unique hibernate id
     */
    @Id
    @GeneratedValue(strategy = GenerationType.AUTO)
    public Long getId() {
        return fId;
    }

    /**
     * Set unique ID of Hibernate.
     * 
     * @param id
     *            the id
     */
    @SuppressWarnings("unused")
    private void setId(Long id) {
        fId = id;
    }

    /**
     * Type code of structure entity. One of {@link EntityType#ATTRIBUTE}, {@link EntityType#METHOD} or
     * {@link EntityType#CLASS}.
     * 
     * @return type code
     */
    public EntityType getType() {
        return fType;
    }

    /**
     * Set type code of structure entity. Must be either {@link EntityType#ATTRIBUTE}, {@link EntityType#CLASS} or
     * {@link EntityType#METHOD}.
     * 
     * @param type
     *            of this entity
     * @throws EvolizerRuntimeException
     *             if illegal type is passed in
     */
    public void setType(EntityType type) {
        if ((type == EntityType.ATTRIBUTE) || (type == EntityType.CLASS) || (type == EntityType.METHOD)) {
            fType = type;
        } else {
            throw new EvolizerRuntimeException("Illegal type " + type + ". Must be ATTRIBUTE, CLASS or METHOD.");
        }
    }

    /**
     * Checks if it's final.
     * 
     * @return true, if this entity is final
     */
    @Transient
    public boolean isFinal() {
        return ChangeModifier.isFinal(fModifiers);
    }

    /**
     * Checks if it's private.
     * 
     * @return true, if this entity is private
     */
    @Transient
    public boolean isPrivate() {
        return ChangeModifier.isPrivate(fModifiers);
    }

    /**
     * Checks if it's protected.
     * 
     * @return true, if this entity is protected
     */
    @Transient
    public boolean isProtected() {
        return ChangeModifier.isProtected(fModifiers);
    }

    /**
     * Checks if it's public.
     * 
     * @return true, if this entity is public
     */
    @Transient
    public boolean isPublic() {
        return ChangeModifier.isPublic(fModifiers);
    }

    /**
     * Sets source code entity's modifiers.
     * 
     * @param modifiers
     *            the modifiers
     */
    public void setModifiers(int modifiers) {
        fModifiers = modifiers;
    }

    /**
     * Returns the modifiers.
     * 
     * @return source code entity's modifiers
     */
    public int getModifiers() {
        return fModifiers;
    }

    /**
     * Returns the unique name.
     * 
     * @return unique name of structure entity.
     */
    @Lob
    public String getUniqueName() {
        return fUniqueName;
    }

    /**
     * Sets unique name of structure entity.
     * 
     * @param uniqueName
     *            the unique name
     */
    public void setUniqueName(String uniqueName) {
        fUniqueName = uniqueName;
    }

    /**
     * Returns the source code changes.
     * 
     * @return {@link List} of {@link SourceCodeChange}s of this structure entity.
     */
    @OneToMany(mappedBy = "rootEntity", cascade = CascadeType.ALL)
    public List<SourceCodeChange> getSourceCodeChanges() {
        return fSourceCodeChanges;
    }

    /**
     * Set {@link List} of {@link SourceCodeChange}s of structure entity.
     * 
     * @param sourceCodeChanges
     *            the source code changes
     */
    public void setSourceCodeChanges(List<SourceCodeChange> sourceCodeChanges) {
        fSourceCodeChanges = sourceCodeChanges;
    }

    /**
     * Add a {@link List} of {@link SourceCodeChange}s to structure entity.
     * 
     * @param sourceCodeChanges
     *            the source code changes
     */
    public void addAllSourceCodeChanges(List<SourceCodeChange> sourceCodeChanges) {
        getSourceCodeChanges().addAll(sourceCodeChanges);
    }

    /**
     * Add a {@link SourceCodeChange} to structure entity.
     * 
     * @param sourceCodeChange
     *            the source code change
     */
    public void addSourceCodeChange(SourceCodeChange sourceCodeChange) {
        getSourceCodeChanges().add(sourceCodeChange);
    }

    /**
     * Returns label for this entity:
     * <ul>
     * <li>In case of a class:
     * 
     * <pre>
     * &lt;className&gt;@&lt;revisionNumber&gt;
     * </pre>
     * 
     * </li>
     * <li>In case of an attribute:
     * 
     * <pre>
     * &lt;attributeName&gt;:&lt;attributeType&gt;@&lt;revisionNumber&gt;
     * </pre>
     * 
     * </li>
     * <li>In case of a method:
     * 
     * <pre>
     * &lt;methodSignature&gt;@&lt;revisionNumber&gt;
     * </pre>
     * 
     * </li>
     * </ul>
     * .
     * 
     * @return label for this entity.
     */
    @Transient
    public String getLabel() {
        String shortName = "";
        switch (getType()) {
            case METHOD:
                int parameterListBegin = getUniqueName().lastIndexOf('(');
                String methodNameWithoutParams = getUniqueName().substring(0, parameterListBegin);
                String methodName = methodNameWithoutParams.substring(methodNameWithoutParams.lastIndexOf('.') + 1);
                String parameterList = getUniqueName().substring(parameterListBegin);
                shortName = methodName + parameterList;
                break;
            default:
                shortName = getUniqueName().substring(getUniqueName().lastIndexOf('.') + 1);
                break;
        }
        return shortName.replace(" ", "") + STRING + (getRevision() == null ? NULL : getRevision().getNumber());
    }

    /**
     * Significance level of structure entity. Value is calculated not saved.
     * 
     * @return significance level
     */
    @Transient
    public Integer getSignificanceLevel() {
        if (fSignificanceLevel < 0) {
            fSignificanceLevel = 0;
            addSigLevels(getSourceCodeChanges());
        }
        return fSignificanceLevel;
    }

    /**
     * Add significance levels of {@link List} of source code changes.
     * 
     * @param changes
     */
    private void addSigLevels(List<SourceCodeChange> changes) {
        for (SourceCodeChange scc : changes) {
            setSignificanceLevel(getSignificanceLevel() + scc.getSignificanceLevel().value());
        }
    }

    /**
     * {@link Revision} of structure entity version.
     * 
     * @return revision of structure entity version
     */
    @ManyToOne
    public Revision getRevision() {
        return fRevision;
    }

    /**
     * Set {@link Revision} of structure entity version.
     * 
     * @param revision
     *            of structure entity version.
     */
    public void setRevision(Revision revision) {
        fRevision = revision;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final int hashCode() {
        return new HashCodeBuilder(11, 19).append(getUniqueName()).append(getRevision()).append(getType()).append(
                getModifiers()).toHashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        StructureEntityVersion other = (StructureEntityVersion) obj;
        return new EqualsBuilder().append(getUniqueName(), other.getUniqueName()).append(getType(), other.getType())
                .append(getRevision(), other.getRevision()).append(getModifiers(), other.getModifiers()).isEquals();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        return getUniqueName() + STRING + (getRevision() == null ? NULL : getRevision().getNumber());
    }

    private void setSignificanceLevel(Integer significanceLevel) {
        fSignificanceLevel = significanceLevel;
    }

    /**
     * {@inheritDoc}
     */
    @Transient
    public String getURI() {
        return null;
    }
}
