/*
 * Copyright 2009 University of Zurich, Switzerland
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.evolizer.changedistiller.model.entities;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.Lob;
import javax.persistence.OneToMany;
import javax.persistence.Transient;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.changedistiller.model.classifiers.SourceRange;
import org.evolizer.core.hibernate.model.api.IEvolizerModelEntity;

/**
 * Source code entity representing one particular AST node.
 * <p>
 * Each source code entity has a {@link EntityType} describing the type of the source code entity.
 * <p>
 * The unique name of a source code entity depends on its {@link EntityType}:
 * <ul>
 * <li>{@link EntityType#ASSERT_STATEMENT}: <code>Expression[<b>:</b>Message]</code></li>
 * <li>{@link EntityType#ATTRIBUTE}: <code>FullyQualifiedName<b>:</b> AttributeType</code></li>
 * <li>{@link EntityType#BREAK_STATEMENT}: <code>[Label]</code></li>
 * <li>{@link EntityType#CATCH_CLAUSE}: <code>ExceptionType</code></li>
 * <li>{@link EntityType#CLASS}: <code>FullyQualifiedName</code></li>
 * <li>{@link EntityType#CONSTRUCTOR_INVOCATION}: <code>Invocation<b>;</b></code></li>
 * <li>{@link EntityType#CONTINUE_STATEMENT}: <code>[Label]</code></li>
 * <li>{@link EntityType#DO_STATEMENT}: <code>Expression</code></li>
 * <li>{@link EntityType#ELSE_STATEMENT}: <code>Expression of if-statement</code></li>
 * <li>{@link EntityType#ENHANCED_FOR_STATEMENT}: <code>Parameter<b>:</b>Expression</code></li>
 * <li>{@link EntityType#EXPRESSION_STATEMENT}: <code>Expression<b>;</b></code></li>
 * <li>{@link EntityType#FOR_STATEMENT}: <code>[Expression]</code></li>
 * <li>{@link EntityType#IF_STATEMENT}: <code>Expression</code></li>
 * <li>{@link EntityType#JAVADOC}: <code>Javadoc as is</code></li>
 * <li>{@link EntityType#LABELED_STATEMENT}: <code>Label</code></li>
 * <li>{@link EntityType#METHOD}: <code>FullyQualifiedSignature</code></li>
 * <li>{@link EntityType#PRIMITIVE_TYPE}: inside a MethodDeclaration
 * <code>[ParameterName|MethodSignature]<b>:</b> TypeName</code>, else: <code>TypeName</code>; method signature is used
 * if the type corresponds to the return type of the method</li>
 * <li>{@link EntityType#RETURN_STATEMENT}: <code>[Expression]</code></li>
 * <li>{@link EntityType#SIMPLE_TYPE}: inside a MethodDeclaration
 * <code>[ParameterName|MethodSignature]<b>:</b> TypeName</code>, else: <code>TypeName</code>; method signature is used
 * if the type corresponds to the return type of the method</li>
 * <li>{@link EntityType#SINGLE_VARIABLE_DECLARATION}: <code>Identifier</code></li>
 * <li>{@link EntityType#SUPER_CONSTRUCTOR_INVOCATION}: <code>Invocation<b>;</b></code></li>
 * <li>{@link EntityType#SWITCH_CASE}: <code>[Expression|<b>default</b>]</code></li>
 * <li>{@link EntityType#SWITCH_STATEMENT}: <code>Expression</code></li>
 * <li>{@link EntityType#SYNCHRONIZED_STATEMENT}: <code>Expression</code></li>
 * <li>{@link EntityType#THEN_STATEMENT}: <code>Expression of if-statement</code></li>
 * <li>{@link EntityType#THROW_STATEMENT}: <code>Expression</code></li>
 * <li>{@link EntityType#TYPE_PARAMETER}: <code>FullyQualifiedName of TypeVariable</code></li>
 * <li>{@link EntityType#VARIABLE_DECLARATION_FRAGMENT}: <code>FullyQualifiedName of Identifier<b>;</b></code></li>
 * <li>{@link EntityType#VARIABLE_DECLARATION_STATEMENT}: <code>Declaration<b>;</b></code></li>
 * <li>{@link EntityType#WHILE_STATEMENT}: <code>Expression</code></li>
 * <li>{@link EntityType#WILDCARD_TYPE}: <code>[<b>extends</b>|<b>super</b>]</code></li>
 * </ul>
 * Entity types that are not mentioned here, have an empty string as unique name.
 * 
 * @author fluri, zubi
 */
@Entity
public class SourceCodeEntity implements IEvolizerModelEntity {

    /**
     * Unique ID, used by Hibernate.
     */
    private Long fId = -1L;

    /**
     * Source code entity's unique name.
     */
    private String fUniqueName;

    /**
     * Type code of source code entity.
     */
    private EntityType fType;

    /**
     * Modifiers of source code entity.
     */
    private Integer fModifiers = 0;

    private List<SourceCodeEntity> fAssociatedEntities = new ArrayList<SourceCodeEntity>();
    private SourceRange fRange = new SourceRange();

    /**
     * Default constructor. Used by Hibernate.
     */
    @SuppressWarnings("unused")
    private SourceCodeEntity() {}

    /**
     * Constructor to initialize a source code entity with a unique name and a type.
     * 
     * @param uniqueName
     *            the name
     * @param type
     *            the type
     * @param range
     *            the range
     */
    public SourceCodeEntity(String uniqueName, EntityType type, SourceRange range) {
        this(uniqueName, type, 0, range);
    }

    /**
     * Constructor to initialize a source code entity with a unique name, a name, a type, and modifiers.
     * 
     * @param uniqueName
     *            the unique name
     * @param type
     *            the type
     * @param modifiers
     *            the modifiers
     * @param range
     *            the range
     */
    public SourceCodeEntity(String uniqueName, EntityType type, Integer modifiers, SourceRange range) {
        setUniqueName(uniqueName);
        setType(type);
        setModifiers(modifiers);
        setSourceRange(range);
    }

    /**
     * Unique ID, used by Hibernate.
     * 
     * @return unique hibernate id
     */
    @Id
    @GeneratedValue(strategy = GenerationType.AUTO)
    public Long getId() {
        return fId;
    }

    /**
     * Set unique ID of Hibernate.
     * 
     * @param id
     *            the id
     */
    @SuppressWarnings("unused")
    private void setId(Long id) {
        fId = id;
    }

    /**
     * Returns the unique name.
     * 
     * @return unique name of source code entity.
     */
    @Lob
    public String getUniqueName() {
        return fUniqueName;
    }

    /**
     * Set unique name of source code entity.
     * 
     * @param uniqueName
     *            the unique name
     */
    public void setUniqueName(String uniqueName) {
        fUniqueName = uniqueName;
    }

    /**
     * Returns the type.
     * 
     * @return source code entity's type code.
     */
    public EntityType getType() {
        return fType;
    }

    /**
     * Sets source code entity's type code.
     * 
     * @param type
     *            the type
     */
    public void setType(EntityType type) {
        fType = type;
    }

    /**
     * Returns the modifiers.
     * 
     * @return source code entity's modifiers
     */
    public int getModifiers() {
        return fModifiers;
    }

    /**
     * Set source code entity's modifiers.
     * 
     * @param modifiers
     *            the modifiers
     */
    public void setModifiers(int modifiers) {
        fModifiers = modifiers;
    }

    /**
     * Checks if it's final.
     * 
     * @return true, if this entity is final
     */
    @Transient
    public boolean isFinal() {
        return ChangeModifier.isFinal(fModifiers);
    }

    /**
     * Checks if it's private.
     * 
     * @return true, if this entity is private
     */
    @Transient
    public boolean isPrivate() {
        return ChangeModifier.isPrivate(fModifiers);
    }

    /**
     * Checks if it's protected.
     * 
     * @return true, if this entity is protected
     */
    @Transient
    public boolean isProtected() {
        return ChangeModifier.isProtected(fModifiers);
    }

    /**
     * Checks if it's public.
     * 
     * @return true, if this entity is public
     */
    @Transient
    public boolean isPublic() {
        return ChangeModifier.isPublic(fModifiers);
    }

    /**
     * Returns the associated entities.
     * 
     * @return the associated entities
     */
    @OneToMany(cascade = CascadeType.ALL)
    @JoinColumn(name = "associated_entity_fk")
    public List<SourceCodeEntity> getAssociatedEntities() {
        return fAssociatedEntities;
    }

    /**
     * Adds an associated entity.
     * 
     * @param entity
     *            the entity to add
     */
    public void addAssociatedEntity(SourceCodeEntity entity) {
        fAssociatedEntities.add(entity);
    }

    /**
     * Sets the associated entities.
     * 
     * @param associatedEntities
     *            the new associated entities
     */
    public void setAssociatedEntities(List<SourceCodeEntity> associatedEntities) {
        fAssociatedEntities = associatedEntities;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37).append(getUniqueName()).append(getType()).append(getModifiers()).append(
                getSourceRange()).append(new ArrayList<SourceCodeEntity>(getAssociatedEntities())).toHashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        SourceCodeEntity other = (SourceCodeEntity) obj;
        return new EqualsBuilder().append(getUniqueName(), other.getUniqueName()).append(getType(), other.getType())
                .append(getModifiers(), other.getModifiers()).append(getSourceRange(), other.getSourceRange())
                .isEquals();
    }

    /**
     * Sets the source range.
     * 
     * @param range
     *            the new source range
     */
    @Transient
    public void setSourceRange(SourceRange range) {
        fRange = range;
    }

    /**
     * Returns the source range.
     * 
     * @return the source range
     */
    @Transient
    public SourceRange getSourceRange() {
        return fRange;
    }

    /**
     * Used for persisting the length by Hibernate.
     * 
     * @return the length
     */
    int getLength() {
        return fRange.getLength();
    }

    /**
     * Sets the length.
     * 
     * @param length
     *            the new length
     */
    @SuppressWarnings("unused")
    private void setLength(int length) {
        fRange.setLength(length);
    }

    /**
     * Used for persisting the offset by Hibernate.
     * 
     * @param offset
     *            the new offset
     */
    void setOffset(int offset) {
        fRange.setOffset(offset);
    }

    /**
     * Returns the offset.
     * 
     * @return the offset
     */
    @SuppressWarnings("unused")
    private int getOffset() {
        return fRange.getOffset();
    }

    /**
     * Returns string representation of this entity, i.e. the type and unique name of it.
     * <p>
     * For debugging purposes only.
     * 
     * @return string representation
     * @see #getType()
     * @see #getUniqueName()
     */
    @Override
    public String toString() {
        return getType().toString() + ": " + getUniqueName();
    }

    /**
     * Returns label of the {@link EntityType} this source code entity represents.
     * 
     * @return label for this entity.
     */
    @Transient
    public String getLabel() {
        return getType().toString();
    }

    /**
     * {@inheritDoc}
     */
    @Transient
    public String getURI() {
        return null;
    }
}
