/*
 * Copyright 2009 University of Zurich, Switzerland
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.evolizer.changedistiller.model.entities;

import javax.persistence.Entity;
import javax.persistence.Transient;

import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.core.exceptions.EvolizerRuntimeException;

/**
 * Method histories store change data about the history of an method.
 * 
 * @author fluri, zubi
 * @see AbstractHistory
 * @see ClassHistory
 * @see AttributeHistory
 */
@Entity
public class MethodHistory extends AbstractHistory {

    /**
     * Default constructor, used by Hibernate.
     */
    @SuppressWarnings("unused")
    private MethodHistory() {}

    /**
     * Instantiates a new method history.
     * 
     * @param method
     *            the method that is added to this history
     */
    public MethodHistory(StructureEntityVersion method) {
        super(method);
    }

    /**
     * Adds method version to this history.
     * 
     * @param version
     *            a method version
     * @throws EvolizerRuntimeException
     *             if version is of any other type than {@link EntityType#METHOD}.
     */
    @Override
    public void addVersion(StructureEntityVersion version) {
        if (version.getType() == EntityType.METHOD) {
            getVersions().add(version);
        } else {
            throw new EvolizerRuntimeException("StructureEntityVersion of type " + version.getType()
                    + " can not be added to MethodHistory.");
        }
    }

    /**
     * Returns label for this history:
     * 
     * <pre>
     * MethodHistory:&lt;methodSignature&gt;
     * </pre>
     * 
     * @return label for this history.
     */
    @Override
    public String getLabel() {
        return super.getLabel();
    }

    /**
     * Returns unique name of this method history, that is
     * 
     * <pre>
     * &lt;fullyQualifiedClassName&gt;.&lt;methodName&gt;(&lt;listOfParamTypes&gt;)
     * </pre>
     * 
     * , e.g. <code>org.foo.Bar.doIt(String, int)</code>.
     * <p>
     * The unique name is built for the most recent method version of this history.
     * 
     * @return unique name of this history.
     */
    @Override
    @Transient
    public String getUniqueName() {
        return super.getUniqueName();
    }
}
