/*
 * Copyright 2009 University of Zurich, Switzerland
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.evolizer.changedistiller.model.entities;

import java.util.HashMap;
import java.util.Map;

import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.MapKey;
import javax.persistence.OneToMany;
import javax.persistence.Transient;

import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.core.exceptions.EvolizerRuntimeException;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;
import org.hibernate.annotations.Cascade;

/**
 * Class histories store change data about the history of a class. It contains its attribute, inner classe, and method
 * histories. Top most element for histories.
 * 
 * @author fluri, zubi
 * @see AbstractHistory
 * @see AttributeHistory
 * @see MethodHistory
 */
@SuppressWarnings("restriction")
@Entity
public class ClassHistory extends AbstractHistory {

    /**
     * {@link Map} of all method histories belonging to this class history. Key is the unique name of method history.
     */
    private Map<String, MethodHistory> fMethodHistories;

    /**
     * {@link Map} of all attribute histories belonging to this class history. Key is the unique name of attribute
     * history.
     */
    private Map<String, AttributeHistory> fAttributeHistories;

    /**
     * {@link Map} of all inner class histories belonging to this class history. Key is the unique name of inner class
     * history.
     */
    private Map<String, ClassHistory> fInnerClassHistories;

    /**
     * Instantiates a new class history and creates a class version for the given unique name and modifiers.
     * 
     * @param uniqueName
     *            the unique name of the class
     * @param modifiers
     *            the modifiers of the class
     * @see ChangeModifier
     */
    public ClassHistory(String uniqueName, int modifiers) {
        initHistories();
        createClass(uniqueName, modifiers);
    }

    /**
     * Default constructor. Initializes all histories and version collections.
     * 
     * @param clazz
     *            the clazz to add to this history
     */
    public ClassHistory(StructureEntityVersion clazz) {
        super(clazz);
        initHistories();
    }

    /**
     * Default constructor, used by Hibernate.
     */
    @SuppressWarnings("unused")
    private ClassHistory() {
        initHistories();
    }

    /**
     * Adds class version to this history.
     * 
     * @param version
     *            a class version
     * @throws EvolizerRuntimeException
     *             if version is of any other type than {@link EntityType#CLASS}.
     */
    @Override
    public void addVersion(StructureEntityVersion version) {
        if (version.getType() == EntityType.CLASS) {
            getVersions().add(version);
        } else {
            throw new EvolizerRuntimeException("StructureEntityVersion of type " + version.getType()
                    + " can not be added to ClassHistory.");
        }
    }

    /**
     * Create an attribute entity with given unique name and modifiers. The attribute is attached to the corresponding
     * attribute history, if exists. Otherwise a new one is created.
     * 
     * @param uniqueName
     *            of attribute entity
     * @param modifiers
     *            of attribute entity
     * 
     * @return newly created attribute entity
     */
    public StructureEntityVersion createAttribute(String uniqueName, int modifiers) {
        AttributeHistory ah = null;
        StructureEntityVersion attribute = new StructureEntityVersion(EntityType.ATTRIBUTE, uniqueName, modifiers);
        if (getAttributeHistories().containsKey(uniqueName)) {
            ah = getAttributeHistories().get(uniqueName);
            ah.addVersion(attribute);
        } else {
            ah = new AttributeHistory(attribute);
            getAttributeHistories().put(uniqueName, ah);
        }
        return attribute;
    }

    /**
     * Create an inner class entity with given unique name and modifiers. An inner class version is added to this
     * resulting Inner{@link ClassHistory}.
     * 
     * @param uniqueName
     *            of class entity
     * @param modifiers
     *            of class entity
     * 
     * @return newly created class entity
     */
    public StructureEntityVersion createInnerClass(String uniqueName, int modifiers) {
        ClassHistory ch = null;
        StructureEntityVersion clazz = new StructureEntityVersion(EntityType.CLASS, uniqueName, modifiers);
        if (getInnerClassHistories().containsKey(uniqueName)) {
            ch = getInnerClassHistories().get(uniqueName);
            ch.addVersion(clazz);
        } else {
            ch = new ClassHistory(clazz);
            getInnerClassHistories().put(uniqueName, ch);
        }
        return clazz;

    }

    /**
     * Create an inner class history for the unique name of the given class version if it does not exist. The history is
     * attached to the class history of this history.
     * 
     * @param clazz
     *            the class version for which a history is to be created
     * 
     * @return class history for the given class version
     */
    public ClassHistory createInnerClassHistory(StructureEntityVersion clazz) {
        ClassHistory ch = null;
        if (getInnerClassHistories().containsKey(clazz.getUniqueName())) {
            ch = getInnerClassHistories().get(clazz.getUniqueName());
        } else {
            ch = new ClassHistory(clazz);
            getInnerClassHistories().put(clazz.getUniqueName(), ch);
        }
        return ch;
    }

    /**
     * Creates a method entity with given unique name and modifiers. The method is attached to the corresponding method
     * history, if exists. Otherwise a new one is created.
     * 
     * @param uniqueName
     *            of method entity
     * @param modifiers
     *            of method entity
     * 
     * @return newly created method entity
     */
    public StructureEntityVersion createMethod(String uniqueName, int modifiers) {
        MethodHistory mh = null;
        StructureEntityVersion method = new StructureEntityVersion(EntityType.METHOD, uniqueName, modifiers);
        if (getMethodHistories().containsKey(uniqueName)) {
            mh = getMethodHistories().get(uniqueName);
            mh.addVersion(method);
        } else {
            mh = new MethodHistory(method);
            getMethodHistories().put(uniqueName, mh);
        }
        return method;
    }

    /**
     * Returns {@link Map} of attribute histories belonging to this class history. Key is the unique name of attribute
     * history.
     * 
     * @return the attribute histories
     */
    @OneToMany
    @MapKey(name = "uniqueName")
    @JoinColumn(name = "attribute_history", nullable = true)
    @Cascade(org.hibernate.annotations.CascadeType.SAVE_UPDATE)
    public Map<String, AttributeHistory> getAttributeHistories() {
        return fAttributeHistories;
    }

    /**
     * Returns class entity with unique name saved as class version with version or creates a version for the class with
     * the given unique name and modifiers.
     * 
     * @param uniqueName
     *            of class entity
     * @param modifiers
     *            the modifiers
     * 
     * @return found or created class entity
     */
    @Transient
    public StructureEntityVersion getClass(String uniqueName, int modifiers) {
        for (int i = 0; i < getVersions().size(); i++) {
            if (getVersions().get(i).getUniqueName().equals(uniqueName) && (getVersions().get(i).getRevision() == null)) {
                return getVersions().get(i);
            }
        }
        return createClass(uniqueName, modifiers);
    }

    /**
     * Returns {@link Map} of inner class histories belonging to this class history. Key is the unique name of inner
     * class history.
     * 
     * @return the inner class histories
     */
    @OneToMany
    @MapKey(name = "uniqueName")
    @JoinColumn(name = "innerclass_history", nullable = true)
    @Cascade(org.hibernate.annotations.CascadeType.SAVE_UPDATE)
    public Map<String, ClassHistory> getInnerClassHistories() {
        return fInnerClassHistories;
    }

    /**
     * Returns label for this history:
     * 
     * <pre>
     * ClassHistory:&lt;className&gt;
     * </pre>
     * 
     * @return label for this history.
     */
    @Override
    public String getLabel() {
        return super.getLabel();
    }

    /**
     * Returns {@link Map} of method histories belonging to this class history. Key is the unique name of method
     * history.
     * 
     * @return the method histories
     */
    @OneToMany
    @MapKey(name = "uniqueName")
    @JoinColumn(name = "method_history", nullable = true)
    @Cascade(org.hibernate.annotations.CascadeType.SAVE_UPDATE)
    public Map<String, MethodHistory> getMethodHistories() {
        return fMethodHistories;
    }

    /**
     * Returns unique name of this class history, that is the fully qualified name of the most recent class version,
     * e.g. <code>org.foo.Bar</code>.
     * 
     * @return unique name of this history.
     */
    @Override
    @Transient
    public String getUniqueName() {
        return super.getUniqueName();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasChanges() {
        boolean hasChanges = false;
        hasChanges |= !getAttributeHistories().isEmpty();
        hasChanges |= !getMethodHistories().isEmpty();
        hasChanges |= !getInnerClassHistories().isEmpty();
        for (int i = 0; !hasChanges && (i < getVersions().size()); i++) {
            hasChanges |= !getVersions().get(i).getSourceCodeChanges().isEmpty();
        }
        // hasChanges |= !getVersions().get(getVersions().size() - 1).getSourceCodeChanges().isEmpty();
        return hasChanges;
    }

    /**
     * Override unique name (key) of attribute history in attribute histories {@link Map}.
     * 
     * @param oldUniqueName
     *            name in {@link Map}
     * @param newUniqueName
     *            new name in {@link Map}
     */
    public void overrideAttributeHistory(String oldUniqueName, String newUniqueName) {
        AttributeHistory tmp = getAttributeHistories().get(oldUniqueName);
        getAttributeHistories().remove(oldUniqueName);
        getAttributeHistories().put(newUniqueName, tmp);
    }

    /**
     * Override unique name (key) of inner class history in inner class histories {@link Map}.
     * 
     * @param oldUniqueName
     *            name in {@link Map}
     * @param newUniqueName
     *            new name in {@link Map}
     */
    public void overrideClassHistory(String oldUniqueName, String newUniqueName) {
        ClassHistory tmp = getInnerClassHistories().get(oldUniqueName);
        getInnerClassHistories().remove(oldUniqueName);
        getInnerClassHistories().put(newUniqueName, tmp);
    }

    /**
     * Override unique name (key) of method history in method histories {@link Map}.
     * 
     * @param oldUniqueName
     *            name in {@link Map}
     * @param newUniqueName
     *            new name in {@link Map}
     */
    public void overrideMethodHistory(String oldUniqueName, String newUniqueName) {
        MethodHistory tmp = getMethodHistories().get(oldUniqueName);
        getMethodHistories().remove(oldUniqueName);
        getMethodHistories().put(newUniqueName, tmp);
    }

    /**
     * Set {@link Map} of attribute histories belonging to this class history.
     * 
     * @param attributeHistories
     *            the attribute histories
     */
    public void setAttributeHistories(Map<String, AttributeHistory> attributeHistories) {
        fAttributeHistories = attributeHistories;
    }

    /**
     * Set {@link Map} of inner class histories belonging to this class history.
     * 
     * @param innerClassHistories
     *            the inner class histories
     */
    public void setInnerClassHistories(Map<String, ClassHistory> innerClassHistories) {
        fInnerClassHistories = innerClassHistories;
    }

    /**
     * Set {@link Map} of method histories belonging to class history.
     * 
     * @param methodHistories
     *            the method histories
     */
    public void setMethodHistories(Map<String, MethodHistory> methodHistories) {
        fMethodHistories = methodHistories;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateLatestVersionWithRevision(Revision revision) {
        // update latest class version
        super.updateLatestVersionWithRevision(revision);

        // update latest method version of each method history
        for (AttributeHistory history : getAttributeHistories().values()) {
            history.updateLatestVersionWithRevision(revision);
        }

        // update inner class histories with with revision
        for (ClassHistory history : getInnerClassHistories().values()) {
            history.updateLatestVersionWithRevision(revision);
        }

        // update latest method version of each method history
        for (MethodHistory history : getMethodHistories().values()) {
            history.updateLatestVersionWithRevision(revision);
        }

    }

    private StructureEntityVersion createClass(String uniqueName, int modifiers) {
        StructureEntityVersion clazz = new StructureEntityVersion(EntityType.CLASS, uniqueName, modifiers);
        addVersion(clazz);
        return clazz;
    }

    private void initHistories() {
        setMethodHistories(new HashMap<String, MethodHistory>());
        setAttributeHistories(new HashMap<String, AttributeHistory>());
        setInnerClassHistories(new HashMap<String, ClassHistory>());
    }

    /**
     * Deletes the provided {@link StructureEntityVersion} from the corresponding {@link MethodHistory}.
     * 
     * <p>
     * If the corresponding history has not any versions left, it is also deleted.
     * 
     * @param method
     *            the structure entity version to delete from the corresponding method history
     */
    public void deleteMethod(StructureEntityVersion method) {
        if ((method.getType() != EntityType.METHOD) || getMethodHistories().isEmpty()) {
            return;
        }
        MethodHistory methodHistory = getMethodHistories().get(method.getUniqueName());
        if ((methodHistory != null) && methodHistory.getVersions().remove(method)) {
            if (methodHistory.getVersions().isEmpty()) {
                getMethodHistories().remove(method.getUniqueName());
            }
        }
    }

    /**
     * Deletes the provided {@link StructureEntityVersion} from the corresponding {@link AttributeHistory}.
     * 
     * <p>
     * If the corresponding history has not any versions left, it is also deleted.
     * 
     * @param attribute
     *            the structure entity version to delete from the corresponding attribute history
     */
    public void deleteAttribute(StructureEntityVersion attribute) {
        if ((attribute.getType() != EntityType.ATTRIBUTE) || getAttributeHistories().isEmpty()) {
            return;
        }
        AttributeHistory attributeHistory = getAttributeHistories().get(attribute.getUniqueName());
        if ((attributeHistory != null) && attributeHistory.getVersions().remove(attribute)) {
            if (attributeHistory.getVersions().isEmpty()) {
                getAttributeHistories().remove(attribute.getUniqueName());
            }
        }
    }
}
