/*
 * Copyright 2009 University of Zurich, Switzerland
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.evolizer.changedistiller.model.entities;

import javax.persistence.Entity;
import javax.persistence.Transient;

import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.core.exceptions.EvolizerRuntimeException;

/**
 * Attribute histories store change data about the history of an attribute.
 * 
 * @author fluri, zubi
 * @see AbstractHistory
 * @see ClassHistory
 */
@Entity
public class AttributeHistory extends AbstractHistory {

    /**
     * Default constructor, used by Hibernate.
     */
    @SuppressWarnings("unused")
    private AttributeHistory() {}

    /**
     * Instantiates a new attribute history.
     * 
     * @param attribute
     *            the attribute that is added to this history
     */
    public AttributeHistory(StructureEntityVersion attribute) {
        super(attribute);
    }

    /**
     * Adds attribute version to this history.
     * 
     * @param version
     *            a attribute version
     * @throws EvolizerRuntimeException
     *             if version is of any other type than {@link EntityType#ATTRIBUTE}.
     */
    @Override
    public void addVersion(StructureEntityVersion version) {
        if (version.getType() == EntityType.ATTRIBUTE) {
            getVersions().add(version);
        } else {
            throw new EvolizerRuntimeException("StructureEntityVersion of type " + version.getType()
                    + " can not be added to AttributeHistory.");
        }
    }

    /**
     * Returns label for this history, that is
     * 
     * <pre>
     * AttributeHistory:&lt;attributeName&gt;:&lt;attributeType&gt;
     * </pre>
     * 
     * @return label for this history.
     */
    @Override
    public String getLabel() {
        return super.getLabel();
    }

    /**
     * Returns unique name of this attribute history, that is
     * 
     * <pre>
     * &lt;fullyQualifiedClassName&gt;.&lt;attributeName&gt; : &lt;attributeType&gt;
     * </pre>
     * 
     * , e.g. <code>org.foo.Bar.imba : String</code>.
     * <p>
     * The unique name is built for the most recent attribute version of this history.
     * 
     * @return unique name of this history.
     */
    @Override
    @Transient
    public String getUniqueName() {
        return super.getUniqueName();
    }
}
