/*
 * Copyright 2009 University of Zurich, Switzerland
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.evolizer.changedistiller.model.entities;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.Lob;
import javax.persistence.OneToMany;
import javax.persistence.Transient;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.evolizer.core.exceptions.EvolizerRuntimeException;
import org.evolizer.core.hibernate.model.api.IEvolizerModelEntity;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;
import org.hibernate.annotations.Cascade;

/**
 * Abstract class for aggregating versions ({@link StructureEntityVersion}) of a source code entity (i.e. a class, a
 * method or an attribute).
 * 
 * @author fluri, zubi
 * @see ClassHistory
 * @see AttributeHistory
 * @see MethodHistory
 */
@SuppressWarnings("restriction")
@Entity
@Inheritance(strategy = InheritanceType.SINGLE_TABLE)
public abstract class AbstractHistory implements IEvolizerModelEntity {

    /**
     * {@link List} of all class versions belonging to this history.
     */
    private List<StructureEntityVersion> fVersions;

    /**
     * Unique ID, used by Hibernate.
     */
    private Long fId = -1L;

    /**
     * Default constructor, used by Hibernate.
     */
    protected AbstractHistory() {
        setVersions(new LinkedList<StructureEntityVersion>());
    }

    /**
     * Constructs history for given version.
     * 
     * @param version
     *            to be added to this history
     */
    public AbstractHistory(StructureEntityVersion version) {
        this();
        addVersion(version);
    }

    /**
     * Unique ID, used by Hibernate.
     * 
     * @return unique Hibernate id
     */
    @Id
    @GeneratedValue(strategy = GenerationType.AUTO)
    public Long getId() {
        return fId;
    }

    /**
     * Set unique ID of Hibernate.
     * 
     * @param id
     *            unique ID
     */
    @SuppressWarnings("unused")
    private void setId(Long id) {
        fId = id;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final int hashCode() {
        return new HashCodeBuilder(11, 19).append(getUniqueName()).append(
                new ArrayList<StructureEntityVersion>(getVersions())).toHashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        AbstractHistory other = (AbstractHistory) obj;
        return new EqualsBuilder().append(getUniqueName(), other.getUniqueName()).append(
                new ArrayList<StructureEntityVersion>(getVersions()),
                new ArrayList<StructureEntityVersion>(other.getVersions())).isEquals();
    }

    /**
     * Returns {@link List} of versions of this history.
     * 
     * @return {@link List} of all versions of this history.
     */
    @OneToMany
    @JoinColumn(name = "history_fk")
    @Cascade(org.hibernate.annotations.CascadeType.SAVE_UPDATE)
    public List<StructureEntityVersion> getVersions() {
        return fVersions;
    }

    /**
     * Set {@link List} of versions of this history.
     * 
     * @param versions
     *            belonging to this history
     */
    public void setVersions(List<StructureEntityVersion> versions) {
        fVersions = versions;
    }

    /**
     * Returns unique name of this history.
     * <p>
     * See subclasses for concrete form.
     * 
     * @return unique name of this history.
     */
    @Lob
    public String getUniqueName() {
        return getVersions().get(getVersions().size() - 1).getUniqueName();
    }

    /**
     * Since unique name is fetched from the most recent attribute version the setter does nothing. It has to be
     * declared for Hibernate convenience.
     * 
     * @param uniqueName
     *            unique name
     */
    protected void setUniqueName(String uniqueName) {}

    /**
     * Returns label for this history.
     * <p>
     * See subclasses for concrete form.
     * 
     * @return label for this history.
     */
    @Transient
    public String getLabel() {
        String label = getVersions().get(getVersions().size() - 1).getLabel();
        label = label.substring(0, label.lastIndexOf("@"));
        return getClass().getSimpleName() + ":" + label;
    }

    /**
     * Since unique name is fetched from the most recent attribute version the setter does nothing. It has to be
     * declared for Hibernate convenience.
     * 
     * @param label
     *            label
     */
    @Transient
    protected void setLabel(String label) {}

    /**
     * {@inheritDoc}
     */
    @Transient
    public String getURI() {
        return null;
    }

    /**
     * Adds version to this history. Version must be of correct type.
     * 
     * @param version
     *            to add
     * @throws EvolizerRuntimeException
     *             if version is the wrong type
     */
    public abstract void addVersion(StructureEntityVersion version);

    /**
     * Checks whether this history contains any changes or not.
     * 
     * @return <code>true</code> if this history contains changes, <code>false</code> otherwise.
     */
    public boolean hasChanges() {
        return !getVersions().isEmpty();
    }

    /**
     * Updates the latest {@link StructureEntityVersion} with the provided revision. This method should be used after
     * one ChangeDistiller pass because it does not set the Revision itself.
     * 
     * @param revision
     *            to be set to the latest structure entity version
     */
    public void updateLatestVersionWithRevision(Revision revision) {
        if (!getVersions().isEmpty()) {
            StructureEntityVersion version = getVersions().get(getVersions().size() - 1);
            if (version.getRevision() == null) {
                version.setRevision(revision);
            }
        }
    }

    /**
     * Returns string representation of this history, i.e. the unique name of it.
     * <p>
     * For debugging purposes only.
     * 
     * @return string representation
     * @see #getUniqueName()
     */
    @Override
    public String toString() {
        return getUniqueName();
    }
}
