package org.evolizer.changedistiller.model.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.List;

import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.changedistiller.model.entities.AbstractHistory;
import org.evolizer.changedistiller.model.entities.AttributeHistory;
import org.evolizer.changedistiller.model.entities.ClassHistory;
import org.evolizer.changedistiller.model.entities.MethodHistory;
import org.evolizer.changedistiller.model.entities.StructureEntityVersion;
import org.evolizer.core.exceptions.EvolizerException;
import org.evolizer.core.exceptions.EvolizerRuntimeException;
import org.evolizer.model.resources.entities.humans.Person;
import org.evolizer.versioncontrol.cvs.model.entities.ModificationReport;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;
import org.junit.Ignore;
import org.junit.Test;

public class HistoryModelTest extends AbstractChangeDistillerModelTest {

    @Test
    public void testAttributeHistoryEquality() throws EvolizerException {
        AttributeHistory history = new AttributeHistory(createAttribute());
        StructureEntityVersion version = createAttribute();
        version.setUniqueName("org.foo.bar.Imba.aValue : Clazz");
        history.addVersion(version);

        List<AttributeHistory> results = saveAndReload(history, "from AttributeHistory", AttributeHistory.class);
        AttributeHistory result = results.get(0);

        checkHistory(history, result, createAttribute(), version);
    }

    private void checkHistory(
            AbstractHistory expected,
            AbstractHistory result,
            StructureEntityVersion first,
            StructureEntityVersion second) throws EvolizerException {
        assertEquals(expected, result);
        assertTrue(expected.hashCode() == result.hashCode());

        assertEquals(2, result.getVersions().size());
        assertTrue(result.getVersions().contains(first));
        assertTrue(result.getVersions().contains(second));

        expected.getVersions().get(0).setUniqueName("w000t");
        assertFalse(expected.equals(result));
    }

    @Test
    public void testClassHistoryEquality() throws EvolizerException {
        ClassHistory history = new ClassHistory(createClass());
        StructureEntityVersion version = createClass();
        version.setUniqueName("fu.bar.Leet");
        history.addVersion(version);

        List<ClassHistory> results = saveAndReload(history, "from ClassHistory", ClassHistory.class);
        ClassHistory result = results.get(0);

        checkHistory(history, result, createClass(), version);
    }

    @Test
    public void testMethodHistoryEquality() throws EvolizerException {
        MethodHistory history = new MethodHistory(createMethod());
        StructureEntityVersion version = createMethod();
        version.setUniqueName("org.leet.Imba.foo(Clazz)");
        history.addVersion(version);

        List<MethodHistory> results = saveAndReload(history, "from MethodHistory", MethodHistory.class);
        MethodHistory result = results.get(0);

        checkHistory(history, result, createMethod(), version);
    }

    @Test(expected = EvolizerRuntimeException.class)
    public void testAddClassVersion() throws EvolizerException {
        AbstractHistory history = new ClassHistory(createClass());
        try {
            history.addVersion(new StructureEntityVersion(EntityType.ATTRIBUTE, null, ChangeModifier.PRIVATE, null));
        } catch (EvolizerRuntimeException e) {
            try {
                history.addVersion(new StructureEntityVersion(EntityType.METHOD, null, ChangeModifier.PUBLIC, null));
            } catch (EvolizerRuntimeException e2) {
                throw e2;
            }

        }
    }

    @Test(expected = EvolizerRuntimeException.class)
    public void testAddMethodVersion() throws EvolizerException {
        AbstractHistory history = new MethodHistory(createMethod());
        try {
            history.addVersion(new StructureEntityVersion(EntityType.ATTRIBUTE, null, ChangeModifier.PRIVATE, null));
        } catch (EvolizerRuntimeException e) {
            try {
                history.addVersion(new StructureEntityVersion(EntityType.CLASS, null, ChangeModifier.PUBLIC, null));
            } catch (EvolizerRuntimeException e2) {
                throw e2;
            }

        }
    }

    @Test(expected = EvolizerRuntimeException.class)
    public void testAddAttributeVersion() throws EvolizerException {
        AbstractHistory history = new AttributeHistory(createAttribute());
        try {
            history.addVersion(new StructureEntityVersion(EntityType.METHOD, null, ChangeModifier.PUBLIC, null));
        } catch (EvolizerRuntimeException e) {
            try {
                history.addVersion(new StructureEntityVersion(EntityType.CLASS, null, ChangeModifier.PUBLIC, null));
            } catch (EvolizerRuntimeException e2) {
                throw e2;
            }

        }
    }

    @Test
    public void testCreateMethod() throws EvolizerException {
        ClassHistory history = new ClassHistory(createClass());
        String uniqueName = "org.foo.Bar.leet(String)";
        assertNull(history.getMethodHistories().get(uniqueName));

        Revision version = createRevision(new Date());
        StructureEntityVersion sev = history.createMethod(uniqueName, ChangeModifier.PUBLIC);
        sev.setRevision(version);
        assertNotNull(history.getMethodHistories().get(uniqueName));

        history = saveAndUniqueReload(history, "from ClassHistory", ClassHistory.class);

        StructureEntityVersion expMethod =
                new StructureEntityVersion(EntityType.METHOD, uniqueName, ChangeModifier.PUBLIC, version);
        assertEquals(expMethod, history.getMethodHistories().get(uniqueName).getVersions().get(0));
    }

    @Test
    public void testCreateAttribute() throws EvolizerException {
        ClassHistory history = new ClassHistory(createClass());
        String uniqueName = "org.foo.Bar.value : String";
        assertNull(history.getAttributeHistories().get(uniqueName));

        Revision version = createRevision(new Date());
        history.addVersion(new StructureEntityVersion(
                EntityType.CLASS,
                "org.test.Imba",
                ChangeModifier.PUBLIC,
                createRevision(new Date())));
        StructureEntityVersion sev = history.createAttribute(uniqueName, ChangeModifier.PRIVATE);
        sev.setRevision(version);
        assertNotNull(history.getAttributeHistories().get(uniqueName));

        history = saveAndUniqueReload(history, "from ClassHistory", ClassHistory.class);

        StructureEntityVersion expMethod =
                new StructureEntityVersion(EntityType.ATTRIBUTE, uniqueName, ChangeModifier.PRIVATE, version);
        assertEquals(expMethod, history.getAttributeHistories().get(uniqueName).getVersions().get(0));
    }

    @Test
    public void testCreateInnerClass() throws EvolizerException {
        ClassHistory history = new ClassHistory(createClass());
        String uniqueName = "org.foo.Bar.value : String";
        assertNull(history.getAttributeHistories().get(uniqueName));

        Revision version = createRevision(new Date());
        history.addVersion(new StructureEntityVersion(
                EntityType.CLASS,
                "org.test.Imba",
                ChangeModifier.PUBLIC,
                createRevision(new Date())));
        StructureEntityVersion sev = history.createAttribute(uniqueName, ChangeModifier.PRIVATE);
        sev.setRevision(version);
        assertNotNull(history.getAttributeHistories().get(uniqueName));

        history = saveAndUniqueReload(history, "from ClassHistory", ClassHistory.class);

        StructureEntityVersion expMethod =
                new StructureEntityVersion(EntityType.ATTRIBUTE, uniqueName, ChangeModifier.PRIVATE, version);
        assertEquals(expMethod, history.getAttributeHistories().get(uniqueName).getVersions().get(0));
    }

    @Test
    public void testcreateInnerClassHistory() throws EvolizerException {
        ClassHistory history = new ClassHistory(createClass());
        String uniqueName = "org.leet.Imba.Foo";
        assertNull(history.getInnerClassHistories().get(uniqueName));
        StructureEntityVersion inner = createClass();
        inner.setUniqueName(uniqueName);
        history.createInnerClassHistory(inner);

        assertNotNull(history.getInnerClassHistories().get(uniqueName));

        List<ClassHistory> results = saveAndReload(history, "from ClassHistory", ClassHistory.class);

        assertEquals(2, results.size());
        assertTrue(results.contains(history));
        ClassHistory innerHistory = new ClassHistory(inner);
        assertTrue(results.contains(innerHistory));
    }

    @Test
    @Ignore
    public void testCascade() {

    }

    @Test
    @Ignore
    public void testGetClass() {

    }

    @Test
    @Ignore
    public void testOverrideClassHistory() {

    }

    private Revision createRevision(Date date) throws EvolizerException {
        Revision rev = new Revision("1.0.0");
        ModificationReport modR = new ModificationReport();
        modR.setAuthor(new Person());
        rev.setReport(modR);
        saveToDB(rev);
        return rev;
    }

    private StructureEntityVersion createAttribute() throws EvolizerException {
        StructureEntityVersion attribute =
                new StructureEntityVersion(
                        EntityType.ATTRIBUTE,
                        "org.foo.bar.Imba.value : Clazz",
                        0,
                        createRevision(new Date()));
        return attribute;
    }

    private StructureEntityVersion createClass() throws EvolizerException {
        StructureEntityVersion clazz =
                new StructureEntityVersion(EntityType.CLASS, "org.foo.bar.Imba", 0, createRevision(new Date()));
        clazz.setRevision(createRevision(new Date()));
        return clazz;
    }

    private StructureEntityVersion createMethod() throws EvolizerException {
        StructureEntityVersion method =
                new StructureEntityVersion(
                        EntityType.METHOD,
                        "org.foo.bar.Imba.foo(String, int)",
                        0,
                        createRevision(new Date()));
        return method;
    }
}