package org.evolizer.changedistiller.model.test;

import static org.junit.Assert.assertEquals;

import org.evolizer.changedistiller.model.classifiers.ChangeModifier;
import org.evolizer.changedistiller.model.classifiers.ChangeType;
import org.evolizer.changedistiller.model.classifiers.EntityType;
import org.evolizer.changedistiller.model.classifiers.SourceRange;
import org.evolizer.changedistiller.model.entities.AbstractHistory;
import org.evolizer.changedistiller.model.entities.AttributeHistory;
import org.evolizer.changedistiller.model.entities.ClassHistory;
import org.evolizer.changedistiller.model.entities.Delete;
import org.evolizer.changedistiller.model.entities.Insert;
import org.evolizer.changedistiller.model.entities.MethodHistory;
import org.evolizer.changedistiller.model.entities.Move;
import org.evolizer.changedistiller.model.entities.SourceCodeChange;
import org.evolizer.changedistiller.model.entities.SourceCodeEntity;
import org.evolizer.changedistiller.model.entities.StructureEntityVersion;
import org.evolizer.changedistiller.model.entities.Update;
import org.evolizer.core.hibernate.model.api.IEvolizerModelEntity;
import org.evolizer.versioncontrol.cvs.model.entities.Revision;
import org.junit.Test;

public class EvolizerModelEntityTest {

    @Test
    public void testHistories() {
        AbstractHistory history = new ClassHistory("test.Imba", ChangeModifier.PUBLIC);
        checkEvolizerModelEntity(history, "ClassHistory:Imba", null);

        history = createAttributeHistory("test.Imba.fName : String");
        checkEvolizerModelEntity(history, "AttributeHistory:fName:String", null);

        history = createMethodHistory("test.Imba.getName()");
        checkEvolizerModelEntity(history, "MethodHistory:getName()", null);

        history = createMethodHistory("test.Imba.setName(String)");
        checkEvolizerModelEntity(history, "MethodHistory:setName(String)", null);
    }

    @Test
    public void testSourceCodeChanges() {
        SourceCodeChange change = createDeleteOperation();
        checkEvolizerModelEntity(change, "STATEMENT_DELETE", null);

        change = createInsertOperation();
        checkEvolizerModelEntity(change, "STATEMENT_INSERT", null);

        change = createMoveOperation();
        checkEvolizerModelEntity(change, "STATEMENT_PARENT_CHANGE", null);

        change = createUpdateOperation();
        checkEvolizerModelEntity(change, "STATEMENT_UPDATE", null);
    }

    private SourceCodeChange createUpdateOperation() {
        SourceCodeEntity updated =
                new SourceCodeEntity("System.out.println(\"test\");", EntityType.METHOD_INVOCATION, new SourceRange(
                        63,
                        80));
        SourceCodeEntity newEntity =
                new SourceCodeEntity("System.out.println(\"aTest\");", EntityType.METHOD_INVOCATION, new SourceRange(
                        63,
                        81));
        SourceCodeEntity parent =
                new SourceCodeEntity("test.Imba.foo(String)", EntityType.METHOD, new SourceRange(38, 178));
        return new Update(ChangeType.STATEMENT_UPDATE, createMethod("test.Imba.foo(String"), updated, newEntity, parent);
    }

    @Test
    public void testSourceCodeEntity() {
        SourceCodeEntity entity =
                new SourceCodeEntity("test.Imba", EntityType.CLASS, ChangeModifier.PUBLIC, new SourceRange(0, 1021));
        checkEvolizerModelEntity(entity, "CLASS", null);
    }

    @Test
    public void testStructureEntityVersion() {
        StructureEntityVersion version =
                new StructureEntityVersion(EntityType.CLASS, "test.Imba", ChangeModifier.PUBLIC);
        checkEvolizerModelEntity(version, "Imba@null", null);

        version.setRevision(new Revision("1.2"));
        checkEvolizerModelEntity(version, "Imba@1.2", null);

        version = new StructureEntityVersion(EntityType.ATTRIBUTE, "test.Imba.fName : String", ChangeModifier.PRIVATE);
        version.setRevision(new Revision("1.2"));
        checkEvolizerModelEntity(version, "fName:String@1.2", null);

        version = new StructureEntityVersion(EntityType.METHOD, "test.Imba.setName(String)", ChangeModifier.PUBLIC);
        version.setRevision(new Revision("1.2"));
        checkEvolizerModelEntity(version, "setName(String)@1.2", null);
    }

    private SourceCodeChange createMoveOperation() {
        SourceCodeEntity moved =
                new SourceCodeEntity("someCall(4);", EntityType.METHOD_INVOCATION, new SourceRange(118, 130));
        SourceCodeEntity newEntity =
                new SourceCodeEntity("someCall(4);", EntityType.METHOD_INVOCATION, new SourceRange(125, 137));
        SourceCodeEntity oldParent =
                new SourceCodeEntity("test.Imba.foo(String)", EntityType.METHOD, new SourceRange(98, 158));
        SourceCodeEntity newParent =
                new SourceCodeEntity("if (isActivated())", EntityType.IF_STATEMENT, new SourceRange(113, 158));
        return new Move(
                ChangeType.STATEMENT_PARENT_CHANGE,
                createMethod("test.Imba.foo(String)"),
                moved,
                newEntity,
                oldParent,
                newParent);
    }

    private SourceCodeChange createInsertOperation() {
        SourceCodeEntity inserted =
                new SourceCodeEntity("fName = name;", EntityType.ASSIGNMENT, new SourceRange(341, 354));
        SourceCodeEntity parent =
                new SourceCodeEntity("test.Imba.getName()", EntityType.METHOD, new SourceRange(311, 380));
        return new Insert(ChangeType.STATEMENT_INSERT, createMethod("test.Imba.setName(String)"), inserted, parent);
    }

    private SourceCodeChange createDeleteOperation() {
        SourceCodeEntity deleted =
                new SourceCodeEntity("System.out.println();", EntityType.METHOD_INVOCATION, new SourceRange(55, 78));
        SourceCodeEntity parent =
                new SourceCodeEntity("test.Imba.getName()", EntityType.METHOD, new SourceRange(25, 114));
        return new Delete(ChangeType.STATEMENT_DELETE, createMethod("test.Imba.getName()"), deleted, parent);
    }

    private StructureEntityVersion createMethod(String uniqueName) {
        return new StructureEntityVersion(EntityType.METHOD, uniqueName, ChangeModifier.PUBLIC);
    }

    private void checkEvolizerModelEntity(IEvolizerModelEntity entity, String label, String uri) {
        assertEquals(label, entity.getLabel());
        assertEquals(uri, entity.getURI());
    }

    private AbstractHistory createMethodHistory(String uniqueName) {
        return new MethodHistory(createMethod(uniqueName));
    }

    private AbstractHistory createAttributeHistory(String uniqueName) {
        StructureEntityVersion attribute =
                new StructureEntityVersion(EntityType.ATTRIBUTE, uniqueName, ChangeModifier.PRIVATE);
        return new AttributeHistory(attribute);
    }

}
