package org.evolizer.changedistiller.model.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import junit.framework.AssertionFailedError;

import org.junit.Test;

/**
 * <p>
 * Base test class to ease the testing of a class's adherence to the <code>equals</code> and <code>hashCode</code>
 * contract. That is whether the equivalence relation is reflexive, symmetric, transitive and consistent (see
 * {@link Object#equals(Object)} and {@link Object#hashCode()}). Furthermore, it is tested whether all relevant
 * non-static and non-final fields are considered for <code>equals</code> and <code>hashCode</code>.
 * </p>
 * <p>
 * To test a class, extend this one and set its generic type to the class under test. Override
 * {@link #createEqualObject()} and {@link #createNotEqualObject()} to provide instances to test equality with.
 * </p>
 * <p>
 * The hook {@link #addTestValues()} can be used to provide test values for user defined types (needed if relevant
 * fields are of such type). Similarly {@link #addFieldTestValues()} can be used to provide test values for a single
 * named field.
 * </p>
 * <p>
 * Fields that are ignored for <code>equals</code> and <code>hashCode</code> can be defined in
 * {@link #getFieldsToExclude()}.
 * </p>
 * 
 * @author zubi
 * @see Object#equals(Object)
 * @see Object#hashCode()
 * @see #createEqualObjects()
 * @see #createNotEqualObject()
 * @see #addTestValues()
 * @see #getFieldsToExclude()
 */
public abstract class AbstractEqualsTest<T> {

    private static final int ITERATION_COUNT = 50;
    private Hashtable<Class<?>, Object[]> fTestValues = null;
    private Hashtable<String, Object[]> fFieldTestValues = null;
    private Class<?> fClassToTest = null;
    private String[] fFieldsToExclude = null;
    private final Object fTestObjB1 = createNotEqualObject();
    private Object[] fAllTestObjects;
    private final Object[] fEqualTestObjects = getEqualObjects();

    public AbstractEqualsTest() {
        setUp();
    }

    private void setUp() {
        fAllTestObjects = new Object[fEqualTestObjects.length + 1];
        System.arraycopy(fEqualTestObjects, 0, fAllTestObjects, 0, fEqualTestObjects.length);
        fAllTestObjects[fAllTestObjects.length - 1] = fTestObjB1;
        fClassToTest = getClassToTest();
        fFieldsToExclude = getFieldsToExclude();
        initTestValues();

        // check successful creation of test objects (not null and different
        // instances)
        assertTrue("at least 3 objects must be provided by createEqualTestInstances()", fEqualTestObjects.length >= 3);
        for (int i = 0; i < fAllTestObjects.length - 1; i++) {
            assertNotNull("createTestInstance() must not return null", fAllTestObjects[i]);
            for (int j = i + 1; j < fAllTestObjects.length; j++) {
                assertNotSame("test objects must be different instances", fAllTestObjects[i], fAllTestObjects[j]);
            }
        }
    }

    private void initTestValues() {
        fFieldTestValues = new Hashtable<String, Object[]>();
        fFieldTestValues.putAll(addFieldTestValues());
        fTestValues = new Hashtable<Class<?>, Object[]>();
        fTestValues.put(String.class, new String[]{"imba", "l33t"});
        fTestValues.put(Boolean.class, new Boolean[]{true, false});
        fTestValues.put(boolean.class, new Boolean[]{true, false});
        fTestValues.put(Byte.class, new Byte[]{new Byte((byte) 0), new Byte((byte) 1)});
        fTestValues.put(byte.class, new Byte[]{new Byte((byte) 0), new Byte((byte) 1)});
        fTestValues.put(Short.class, new Short[]{new Short((short) 0), new Short((short) 1)});
        fTestValues.put(short.class, new Short[]{new Short((short) 0), new Short((short) 1)});
        fTestValues.put(Integer.class, new Integer[]{new Integer(0), new Integer(1)});
        fTestValues.put(int.class, new Integer[]{new Integer(0), new Integer(1)});
        fTestValues.put(Long.class, new Long[]{new Long(0L), new Long(1L)});
        fTestValues.put(long.class, new Long[]{new Long(0L), new Long(1L)});
        fTestValues.put(Float.class, new Float[]{new Float(0F), new Float(1F)});
        fTestValues.put(float.class, new Float[]{new Float(0F), new Float(1F)});
        fTestValues.put(Double.class, new Double[]{new Double(0D), new Double(1D)});
        fTestValues.put(double.class, new Double[]{new Double(0D), new Double(1D)});
        fTestValues.put(Character.class, new Character[]{new Character('a'), new Character('b')});
        fTestValues.put(char.class, new Character[]{new Character('a'), new Character('b')});
        fTestValues.putAll(addTestValues());
    }

    /**
     * First look for test values for a specific field. If nothing found, look for its type.
     */
    private Object getTestValue(Field field, int index) {
        Object[] values = fFieldTestValues.get(field.getName());

        if (values == null) {
            values = fTestValues.get(field.getType());
            if (values == null) {
                if (field.getType().isEnum()) {
                    values = field.getType().getEnumConstants();
                }
                if ((values == null) || (values.length <= index)) {
                    fail("Unable to find test value for class '" + field.getName()
                            + "'. Provide the value by overwriting 'addTestValues()'.");
                }
            }
        }
        return values[index];
    }

    /**
     * <p>
     * Tests whether <code>equals</code> and <code>hashCode</code> consider all relevant fields of the class being
     * tested.
     * </p>
     * <p>
     * It uses {@link Field#setAccessible(boolean)} to gain access to the fields to initialize them with test values.
     * </p>
     * <p>
     * Static and final fields are ignored.
     * </p>
     * 
     * @see #addTestValues()
     * @see #getFieldsToExclude()
     */
    @Test
    public void testRelevantFields() {
        Object obj1, obj2;
        try {
            Constructor<?> constructor = fClassToTest.getDeclaredConstructor((Class[]) null);
            constructor.setAccessible(true);
            obj1 = constructor.newInstance((Object[]) null);
            obj2 = constructor.newInstance((Object[]) null);

            List<Field> fields = getRelevantFields();
            for (Field field : fields) {
                field.setAccessible(true);
                field.set(obj1, getTestValue(field, 0));
                field.set(obj2, getTestValue(field, 0));
            }
            assertEquals("Instances initialized with same values must be equal.", obj1, obj2);
            assertTrue("Hash values of equal instances must be the same.", obj1.hashCode() == obj2.hashCode());

            for (Field field : fields) {
                field.set(obj1, getTestValue(field, 1));
                assertFalse("Single relevant field '" + field.getName() + "' changed, objects must not be equal.", obj1
                        .equals(obj2));
                assertFalse("Single relevant field  '" + field.getName()
                        + "' changed, hash code of objects must not be the same.", obj1.hashCode() == obj2.hashCode());
                field.set(obj1, getTestValue(field, 0));
            }
        } catch (InstantiationException e) {
            e.printStackTrace();
            throw new AssertionFailedError("Unable to create instance of class " + fClassToTest.getName());
        } catch (IllegalAccessException e) {
            e.printStackTrace();
            throw new AssertionFailedError("Unable to create instance of class " + fClassToTest.getName());
        } catch (SecurityException e) {
            e.printStackTrace();
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
    }

    private void getDeclaredField(Class<?> clazz, List<Field> fields) {
        if (clazz.getSuperclass() != null) {
            fields.addAll(Arrays.asList(clazz.getDeclaredFields()));
            getDeclaredField(clazz.getSuperclass(), fields);
        }
    }

    private List<Field> getRelevantFields() throws IllegalAccessException {
        ArrayList<Field> fields = new ArrayList<Field>();
        // if (Modifier.isAbstract(fClassToTest.getModifiers())) {
        getDeclaredField(fClassToTest, fields);
        // }
        List<String> excludes = Arrays.asList(fFieldsToExclude);

        for (Iterator<Field> iterator = fields.iterator(); iterator.hasNext();) {
            Field field = iterator.next();
            if (excludes.contains(field.getName()) || Modifier.isFinal(field.getModifiers())
                    || Modifier.isStatic(field.getModifiers())) {
                iterator.remove();
            }
        }
        return fields;
    }

    /**
     * Hook to provide test values for user defined types. These values must not be equal. They are used for every field
     * of the given type and for which no specific test values are provided (see {@link #addFieldTestValues()}.
     * 
     * @return {@link Map} containing type and array of concrete test values that are not equal
     */
    protected Map<Class<?>, Object[]> addTestValues() {
        return Collections.emptyMap();
    }

    /**
     * Hook to provide test values for a specific named field. These values are first considered before searching for
     * test values of a specific type (see {@link #addTestValues()}).
     * <p>
     * Use this hook for providing test values for fields of generic types.
     * </p>
     * 
     * @return {@link Map} containing name of field and an array of concrete test values that are not equal
     */
    protected Map<String, Object[]> addFieldTestValues() {
        return Collections.emptyMap();
    }

    private Class<?> getClassToTest() {
        ParameterizedType parameterizedType = (ParameterizedType) getClass().getGenericSuperclass();
        return (Class<?>) parameterizedType.getActualTypeArguments()[0];
    }

    /**
     * Hook to define fields which are not taken into account for equality.
     * 
     * @return array of field names to exclude
     */
    protected String[] getFieldsToExclude() {
        return new String[0];
    }

    /**
     * <p>
     * Hook to create and return objects that are supposed to be equal.
     * </p>
     * <p>
     * At least three objects must be returned. If there are less than three objects returned, the method is called
     * three times and the object at <code>index == 0</code> is used for equality testing. The test objects must not be
     * <code>null</code> and must not be identical references.
     * </p>
     * 
     * @return an array of at least 3 objects that are supposed to be equal, i.e. each object in the array should
     *         compare equal to each other
     */
    protected abstract T[] createEqualObjects();

    private Object[] getEqualObjects() {
        Object[] testObjects = createEqualObjects();
        if (testObjects != null) {
            if (testObjects.length < 3) {
                testObjects = new Object[]{createEqualObjects()[0], createEqualObjects()[0], createEqualObjects()[0]};
            }
        }
        return testObjects;
    }

    /**
     * Creates and returns an instance of the class under test.
     * 
     * @return a new instance of the class under test; each object returned from this method should compare equal to
     *         each other, but not to the objects returned from {@link #createEqualObjects() createInstance}.
     */
    protected abstract T createNotEqualObject();

    /**
     * Tests whether <code>equals</code> holds up against a new <code>Object</code> (should always be <code>false</code>
     * ).
     */
    @Test
    public final void testEqualsAgainstNewObject() {
        Object newObj = new Object();
        for (Object testObject : fAllTestObjects) {
            assertFalse("test objects must not be equal to new Object()", newObj.equals(testObject));
        }
    }

    /**
     * Tests for non-null reference value x, i.e. <code>x.equals(null)</code> should return <code>false</code>.
     */
    @Test
    public final void testNullReference() {
        for (Object testObj : fAllTestObjects) {
            assertFalse("testObject.equals(null) should return false", testObj.equals(null));
        }
    }

    /**
     * Tests whether <code>equals</code> holds up against objects that should not compare equal.
     */
    @Test
    public final void testNotEqual() {
        for (Object testObject : fEqualTestObjects) {
            assertFalse(
                    "test objects provided by createEqualTestInstances() are supposed to be not equal to the object provided by createNotEqualObject()",
                    testObject.equals(fTestObjB1));
        }
    }

    /**
     * Tests whether <code>equals</code> is <em>consistent</em>.
     */
    @Test
    public final void testConsistent() {
        for (int i = 0; i < ITERATION_COUNT; ++i) {
            testEqualsAgainstNewObject();
            testNullReference();
            testReflexive();
            testTransitive();
            testNotEqual();
        }
    }

    /**
     * Tests whether <code>equals</code> is <em>reflexive</em>, i.e. <code>x.equals(x)</code> should return
     * <code>true</code>.
     */
    @Test
    public final void testReflexive() {
        for (Object testObj : fAllTestObjects) {
            assertEquals("testObject.equals(testObject) should return true", testObj, testObj);
        }
    }

    /**
     * Tests whether <code>equals</code> is <em>symmetric</em>, i.e. <code>x.equals(y) == y.equals(x)</code> should
     * return <code>true</code>.
     */
    @Test
    public final void testSymmetric() {
        String rationale = "x.equals(y) == y.equals(x) should hold";

        for (int i = 0; i < fEqualTestObjects.length - 1; i++) {
            for (int j = i + 1; j < fEqualTestObjects.length; j++) {
                assertEquals(rationale, fEqualTestObjects[i], fEqualTestObjects[j]);
                assertEquals(rationale, fEqualTestObjects[j], fEqualTestObjects[i]);
            }
        }
    }

    /**
     * Tests whether <code>equals</code> is <em>transitive</em>, i.e.
     * <code>x.equals(y) == y.equals(z) == x.equals(z) should hold</code> should return <code>true</code>.
     */
    @Test
    public final void testTransitive() {
        String rationale = "x.equals(y) == y.equals(z) == x.equals(z) should hold";
        for (int i = 0; i < fEqualTestObjects.length - 2; i++) {
            for (int j = i + 1; j < fEqualTestObjects.length - 1; j++) {
                assertEquals(rationale, fEqualTestObjects[i], fEqualTestObjects[j]);
                assertEquals(rationale, fEqualTestObjects[j], fEqualTestObjects[j + 1]);
                assertEquals(rationale, fEqualTestObjects[i], fEqualTestObjects[j + 1]);
            }
        }
    }

    /**
     * Tests the <code>hashCode</code> contract.
     */
    @Test
    public final void testHashCodeContract() {
        String rationale = "hashCode of equal objects must be the same";
        for (int i = 0; i < fEqualTestObjects.length - 2; i++) {
            for (int j = i + 1; j < fEqualTestObjects.length - 1; j++) {
                assertTrue(rationale, fEqualTestObjects[i].hashCode() == fEqualTestObjects[j].hashCode());
                assertTrue(rationale, fEqualTestObjects[j].hashCode() == fEqualTestObjects[j + 1].hashCode());
                assertTrue(rationale, fEqualTestObjects[i].hashCode() == fEqualTestObjects[j + 1].hashCode());
            }
        }
    }

    /**
     * Tests the consistency of <code>hashCode</code>.
     */
    @Test
    public final void testHashCodeIsConsistentAcrossInvocations() {
        int[] hashCodes = new int[fAllTestObjects.length];
        for (int i = 0; i < fAllTestObjects.length; i++) {
            hashCodes[i] = fAllTestObjects[i].hashCode();
        }

        String rationale = "hashCode must be consistent over several invocations";
        for (int i = 0; i < ITERATION_COUNT; ++i) {
            for (int j = 0; j < hashCodes.length; j++) {
                assertTrue(rationale, fAllTestObjects[j].hashCode() == hashCodes[j]);
            }
        }
    }
}