package org.evolizer.changedistiller.hibernate;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.evolizer.core.exceptions.EvolizerException;
import org.evolizer.core.hibernate.session.api.IEvolizerSession;
import org.evolizer.core.preferences.EvolizerPreferences;

public class ChangeDistillerSessionFactory {
	private final static Logger logger = ChangeDistillerHibernatePlugin.getLogManager().getLogger(ChangeDistillerSessionFactory.class.getName());
	
	/**
	 * Returns an implementation of {@link IEvolizerSession} configured using the persistent properties of the passed {@link IProject}.
	 * 
	 * @param project to gain database configuration
	 * @return an implementation of IEvolizerSession
	 * 
	 * @throws EvolizerException if session could not be created because preference store of project is not accessible.
	 * 							 Reasons include
	 * 							 <ul>
	 * 								<li> Project does not exist.</li>
	 * 								<li> Project is not local.</li>
	 * 								<li> Project is not open.</li>
	 * 							 </ul>
	 */
	public static ChangeDistillerSessionImpl getEvolizerSession(IProject project) throws EvolizerException {
		logger.debug("getEvolizerSession() has been invoked for project '" + project.getName() + "'");
		
		try {
			String host = project.getPersistentProperty(EvolizerPreferences.DB_HOST);
			String database = project.getPersistentProperty(EvolizerPreferences.DB_NAME);
			
			String user = project.getPersistentProperty(EvolizerPreferences.DB_USER);
			String password = project.getPersistentProperty(EvolizerPreferences.DB_PASSWORD);
			
			return ChangeDistillerSessionFactory.getEvolizerSession(
					host + "/" + database,
					"org.hibernate.dialect.MySQLDialect",
					"com.mysql.jdbc.Driver",
					user,
					password
			);
			
		} catch (CoreException e) {
			logger.error("Error while fetching persistent properties from project '" + project.getName() + "'." + e.getMessage(), e);
			throw new EvolizerException(e);
		}
	}
	
	/**
	 * Returns an implementation of {@link IEvolizerSession} configured with the passed params.
	 * 
	 * @param dbUrl database host (e.g. mysql://localhost:3306/evolizer_test)
	 * @param dbDialect database dialect (e.g. org.hibernate.dialect.MySQLDialect)
	 * @param dbDriverName jdbc-compliant database driver (e.g. com.mysql.jdbc.Driver)
	 * @param dbUser database username
	 * @param dbPasswd database password for dbUser
	 *
	 * @return an implementation of IEvolizerSession
	 */
	public static ChangeDistillerSessionImpl getEvolizerSession(String dbUrl,
			   										  String dbDialect,
			   										  String dbDriverName,
			   										  String dbUser,
			   										  String dbPasswd){
		logger.debug("getEvolizerSession() has been invoked with params '" + 
					dbUrl + "', '" +
					dbDialect + "', '" +
					dbDriverName + "', '" +
					dbUser + "', '" +
					dbPasswd + "'");
		
		return new ChangeDistillerSessionImpl(dbUrl,
				   dbDialect,
				   dbDriverName,
				   dbUser,
				   dbPasswd);
	}
}
