package ifi.dbtg.ldbe.svd;

import ifi.dbtg.ldbe.algebra.IndexOutOfMatrixBoundariesException;
import ifi.dbtg.ldbe.algebra.Matrix;
import ifi.dbtg.ldbe.algebra.NotMatchingSizeException;
import ifi.dbtg.ldbe.algebra.Vector;

/**
 * Implementation of Singular Value Decomporsition
 * 
 * @author Michal@Mourad
 * 
 */
public class SVD {
	private final double tol = Double.MIN_NORMAL / EPSILON;
	/**
	 * Smallest positive number such that 1 - EPSILON is not numerically equal
	 * to 1.
	 */
	public static final double EPSILON = 0x1.0p-53;
	Matrix A;
	Matrix U;
	Matrix V;
	Matrix S;

	/**
	 * public constructor
	 * 
	 * @param m
	 * @throws IndexOutOfMatrixBoundariesException
	 * @throws NotMatchingSizeException
	 */
	public SVD(Matrix m) throws IndexOutOfMatrixBoundariesException,
			NotMatchingSizeException {
		A = m.clone();
		double[] w = new double[m.colNum()];
		double[][] u = m.getData();
		double[][] v = new double[m.colNum()][m.colNum()];

		this.svd(EPSILON, m.rowNum(), m.colNum(), u, v, w);

		this.U = new Matrix(u);
		this.V = new Matrix(v);

		int count = 0;
		for (double d : w) {
			if (d != 0)
				count++;
		}
		this.S = Matrix.getDiagonalMatrix(w);

	}

	/**
	 * returns i-th dimension product u_i*s_i*{v_i}^T
	 * 
	 * @param i
	 * @return
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	public Matrix getDimensionMatrix(int i)
			throws IndexOutOfMatrixBoundariesException {
		return Vector.multiply(
				this.getU().getColumn(i),
				Vector.multiply(this.getV().getColumn(i),
						this.getS().getEntry(i, i)));
	}

	/**
	 * \ returns Frobenius norm of the matrix being decomposed
	 * 
	 * @return
	 */
	public double getFrobeniusNorm() {
		double sum = 0.0;
		for (int i = 0; i < this.getS().rowNum(); i++) {
			sum += Math.pow(this.getS().getEntry(i, i), 2.0);
		}
		return Math.sqrt(sum);
	}

	public Matrix getU() {
		return this.U;
	}

	public Matrix getS() {
		return this.S;
	}

	public Matrix getVT() {
		return this.V.transpose();
	}

	public Matrix getV() {
		return this.V;
	}

	/*
	 * Singular Value Decompsotion calculations. Derived from Algol code of G.H
	 * Golub and C. Reinsch, 'Singular Value Decomposition and Least Squares
	 * Solutions' (Handbook Series Linear Algebra'
	 */
	void svd(double eps, int m, int n, double[][] u, double[][] v, double[] q) {
		int i, j, k, l = 0, ll;
		double c, f, g, h, s, x, y, z;
		double e[] = new double[n];

		// Householder's reduction to bidiagonal form;

		g = x = 0.0;
		for (i = 0; i < n; i++) {
			e[i] = g;
			s = 0.0;
			l = i + 1;

			for (j = i; j < m; j++)
				s += u[j][i] * u[j][i];
			if (s < tol)
				g = 0.0;
			else {
				f = u[i][i];
				g = (f < 0.0 ? Math.sqrt(s) : -Math.sqrt(s));
				h = f * g - s;
				u[i][i] = f - g;
				for (j = l; j < n; j++) {
					s = 0.0;
					for (k = i; k < m; k++)
						s += u[k][i] * u[k][j];
					f = s / h;
					for (k = i; k < m; k++)
						u[k][j] += f * u[k][i];
				}
			}
			q[i] = g;
			s = 0.0;

			for (j = l; j < n; j++)
				s += u[i][j] * u[i][j];
			if (s < tol)
				g = 0.0;
			else {
				f = u[i][i + 1];
				g = (f < 0 ? Math.sqrt(s) : -Math.sqrt(s));
				h = f * g - s;
				u[i][i + 1] = f - g;
				for (j = l; j < n; j++)
					e[j] = u[i][j] / h;
				for (j = l; j < m; j++) {
					s = 0.0;
					for (k = l; k < n; k++)
						s += u[j][k] * u[i][k];
					for (k = l; k < n; k++)
						u[j][k] += s * e[k];
				}
			}
			y = Math.abs(q[i]) + Math.abs(e[i]);
			if (y > x)
				x = y;
		}

		// accumulation of right-hand transformations;
		for (i = n - 1; i >= 0; i--) {
			if (g != 0) {
				h = u[i][i + 1] * g;
				for (j = l; j < n; j++)
					v[j][i] = u[i][j] / h;
				for (j = l; j < n; j++) {
					s = 0.0;
					for (k = l; k < n; k++)
						s += u[i][k] * v[k][j];
					for (k = l; k < n; k++)
						v[k][j] += s * v[k][i];
				}
			}
			for (j = l; j < n; j++)
				v[i][j] = v[j][i] = 0.0;
			v[i][i] = 1.0;
			g = e[i];
			l = i;
		}

		// accumulation of left-hand transformations;
		for (i = n - 1; i >= 0; i--) {
			l = i + 1;
			g = q[i];
			for (j = l; j < n; j++)
				u[i][j] = 0.0;
			if (g != 0) {
				h = u[i][i] * g;
				for (j = l; j < n; j++) {
					s = 0.0;
					for (k = l; k < m; k++)
						s += u[k][i] * u[k][j];
					f = s / h;
					for (k = i; k < m; k++)
						u[k][j] += f * u[k][i];
				}
				for (j = i; j < m; j++)
					u[j][i] /= g;
			} else
				for (j = i; j < m; j++)
					u[j][i] = 0.0;

			u[i][i] = u[i][i] + 1.0;
		}
		// diagonalization of tile bidiagonal form;
		eps *= x;
		for (k = n - 1; k >= 0; k--) {
			test_f_splitting: do {// TEST F SPLITTING: //
				goto_test_f_convergence: do {
					for (l = k; l >= 0; l--) {
						if (Math.abs(e[l]) <= eps)
							break goto_test_f_convergence; // GOTO test f
						// convergence;
						if (Math.abs(q[l - 1]) <= eps)
							break;// GOTO cancellation
					}

					// cancellation of e[l] if l>1;
					c = 0.0; // CANCELLATION:
					s = 1.0;
					ll = l - 1;
					for (i = l; i <= k; i++) {
						f = s * e[i];
						e[i] *= c;
						if (Math.abs(f) <= eps)
							break goto_test_f_convergence;// GOTO test f
						// convergence;
						g = q[i];
						h = q[i] = Math.sqrt(f * f + g * g);
						c = g / h;
						s = -f / h;
						for (j = 0; j < m; j++) {
							y = u[j][ll];
							z = u[j][i];
							u[j][ll] = y * c + z * s;
							u[j][i] = -y * s + z * c;
						}
					}
					break goto_test_f_convergence;
				} while (true);
				// TEST F CONVERGENCE:
				z = q[k];
				if (l == k)
					break test_f_splitting;// GOTO convergence;

				// shift from bottom 2x2 minor;

				x = q[l];
				y = q[k - 1];
				g = e[k - 1];
				h = e[k];
				f = ((y - z) * (y + z) + (g - h) * (g + h)) / (2 * h * y);
				g = Math.sqrt(f * f + 1);
				f = ((x - z) * (x + z) + h * (y / (f < 0 ? f - g : f + g) - h))
						/ x;

				// next QR transformation
				c = s = 1.0;
				for (i = l + 1; i <= k; i++) {
					g = e[i];
					y = q[i];
					h = s * g;
					g *= c;
					e[i - 1] = z = Math.sqrt(f * f + h * h);
					c = f / z;
					s = h / z;
					f = x * c + g * s;
					g = -x * s + g * c;
					h = y * s;
					y *= c;
					for (j = 0; j < n; j++) {
						x = v[j][i - 1];
						z = v[j][i];
						v[j][i - 1] = x * c + z * s;
						v[j][i] = -x * s + z * c;
					}
					q[i - 1] = z = Math.sqrt(f * f + h * h);
					c = f / z;
					s = h / z;
					f = c * g + s * y;
					x = -s * g + c * y;
					for (j = 0; j < m; j++) {
						y = u[j][i - 1];
						z = u[j][i];
						u[j][i - 1] = y * c + z * s;
						u[j][i] = -y * s + z * c;
					}
				}
				e[l] = 0.0;
				e[k] = f;
				q[k] = x;

			} while (true); // GOTO test f splitting; //TODO

			// CONVERGENCE: //TODO
			if (z < 0) {
				// q[k] is made non-negative;
				q[k] = -z;
				for (j = 0; j < n; j++)
					v[j][k] = -v[j][k];
			}
		}

	}

}
