package ifi.dbtg.ldbe.svd;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import oracle.jdbc.*;
import oracle.jdbc.internal.OraclePreparedStatement;

import org.apache.log4j.Logger;

import ifi.dbtg.ldbe.algebra.IndexOutOfMatrixBoundariesException;
import ifi.dbtg.ldbe.algebra.Matrix;
import ifi.dbtg.ldbe.algebra.NotMatchingSizeException;
import ifi.dbtg.ldbe.db.DBConnection;
import ifi.dbtg.ldbe.misc.InterpolationException;
import ifi.dbtg.ldbe.ts.RecoveryMethod;
import ifi.dbtg.ldbe.ts.TSSet;
import ifi.dbtg.ldbe.ts.TimeSeries;

/**
 * Iterative Singular Value Decomposition method implementation
 * 
 * @author Michal@Mourad
 * 
 */
public class IterativeSVD {
	class SVandIndex implements Comparable {
		int index;
		Double value;

		SVandIndex(int index, double value) {
			this.index = index;
			this.value = value;
		}

		public String toString() {
			return index + ": " + value;
		}

		@Override
		public int compareTo(Object arg0) {
			return this.value.compareTo(((SVandIndex) arg0).value);

		}

	}

	final static Logger logger = Logger.getLogger(IterativeSVD.class);
	public static String decimalFormat = "#0.000";

	// file: '2010.11.18 [05] SEST TS queries.sql' ; query: '08 - update SVD'
	private final static String SQLQueryUpdSVD = "UPDATE missing_value SET val05=:val,info05_01=:info1,info05_02=:info2 WHERE series_id=:id AND ts=:ts AND norm=:norm";
	private final static String SQLQueryAlignRes = "INSERT INTO run_time VALUES(:series_id,'SVD',:exec_time,:no_iter,:ref_no)";
	private final static double threshold = 10E-5;
	private final static double max_iter = 50;

	private TimeSeries ts;
	private List<TimeSeries> mostCorrelated;
	private Object[] objects;
	private Matrix X;
	private Integer[] ts_mvIndexes;

	Integer NO_OF_DIM_KEPT;

	/**
	 * loads all necessary data from DB needed for iterative approximation
	 * 
	 * @throws SQLException
	 * @throws InterpolationException
	 * @throws NotMatchingSizeException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	private void createMatrix0() throws SQLException, InterpolationException,
			NotMatchingSizeException, IndexOutOfMatrixBoundariesException {
		Double[] ts_data = ts
				.getFullData(0L, RecoveryMethod.LIN_INTERP_EVENTS);
		this.ts_mvIndexes = ts.getMVindexes();

		Double[][] mostCorrelatedData = new Double[mostCorrelated.size()][];

		for (int i = 0; i < mostCorrelatedData.length; i++) {
			mostCorrelatedData[i] = mostCorrelated.get(i).getFullData(0,
					RecoveryMethod.LIN_INTERP_EVENTS);
		}

		this.X = new Matrix(ts.getLength(), mostCorrelated.size() + 1);

		for (int i = 0; i < ts_data.length; i++) {
			this.X.setEntry(i, 0, ts_data[i]);
			for (int j = 0; j < mostCorrelated.size(); j++) {
				this.X.setEntry(i, j + 1, mostCorrelatedData[j][i]);
			}
		}
	}

	/**
	 * loads all necessary data from DB needed for iterative approximation
	 * 
	 * @throws SQLException
	 * @throws InterpolationException
	 * @throws NotMatchingSizeException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	private void createMatrix1() throws SQLException, InterpolationException,
			NotMatchingSizeException, IndexOutOfMatrixBoundariesException {
		Double[] ts_data = ts
				.getFullData(0L, RecoveryMethod.LIN_INTERP_EVENTS);
		this.ts_mvIndexes = ts.getMVindexes();

		Double[][] mostCorrelatedData = new Double[1][];

		mostCorrelatedData[0] = mostCorrelated.get(ts.getCat02() - 1)
				.getFullData(0, RecoveryMethod.LIN_INTERP_EVENTS);

		this.X = new Matrix(ts.getLength(), mostCorrelated.size() + 1);

		for (int i = 0; i < ts_data.length; i++) {
			this.X.setEntry(i, 0, ts_data[i]);
			for (int j = 0; j < mostCorrelatedData.length; j++) {
				this.X.setEntry(i, j + 1, mostCorrelatedData[j][i]);
			}
		}
	}

	/**
	 * constructor
	 * 
	 * @param ts
	 * @throws SQLException
	 * @throws InterpolationException
	 * @throws NotMatchingSizeException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	public IterativeSVD(TimeSeries ts) throws SQLException,
			InterpolationException, NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {
		this.ts = ts;
		this.mostCorrelated = ts.getMostCorrel(RecoveryMethod.SVD,
				RecoveryMethod.LIN_INTERP_EVENTS);

		if (ts.getCat04() == 0) {
			this.createMatrix0();
			this.process0();
			this.save();
		}
		if (ts.getCat04() == 1) {
			this.createMatrix1();
			this.process1();
			this.save();
		}

	}

	/**
	 * saves result of approximation to MISSING_VALUE table
	 * 
	 * @throws SQLException
	 */
	private void save() throws SQLException {
		String aux = "";

		for (TimeSeries ts : mostCorrelated)
			aux = aux + ts.getId() + " ";

		Connection conn = DBConnection.getConnection();
		conn.setAutoCommit(false);

		OraclePreparedStatement SQLStmt = (OraclePreparedStatement) conn
				.prepareStatement(SQLQueryUpdSVD);
		SQLStmt.setIntAtName("id", ts.getId());
		SQLStmt.setIntAtName("norm", ts.getNormMthd().ordinal());
		SQLStmt.setStringAtName("info1", aux);
		SQLStmt.setStringAtName("info2", NO_OF_DIM_KEPT.toString());

		for (int i = 0; i < ts_mvIndexes.length; i++) {
			SQLStmt.setIntAtName("ts", ts_mvIndexes[i]);
			SQLStmt.setDoubleAtName("val",
					X.getEntry(ts_mvIndexes[i] / ts.getGranularity(), 0));
			SQLStmt.executeUpdate();
		}
		SQLStmt.close();

		if (!ts.isAligned(RecoveryMethod.SVD)) {
			ts.setAlignmentCode(ts.getAlignmentCode()
					+ RecoveryMethod.getDeterminator(RecoveryMethod.SVD));
			ts.updateAlign(conn, ts.getAlignmentCode());
		}

		conn.commit();
		conn.setAutoCommit(true);
	}

	/**
	 * saves execution time to RUN_TIME table
	 * 
	 * @param execTime
	 * @param iter
	 * @throws SQLException
	 */
	private void saveTime(long execTime, int iter) throws SQLException {
		String aux = "";

		for (TimeSeries ts : mostCorrelated)
			aux = aux + ts.getId() + " ";

		Connection conn = DBConnection.getConnection();
		conn.setAutoCommit(false);

		OraclePreparedStatement SQLStmt = (OraclePreparedStatement) conn
				.prepareStatement(SQLQueryAlignRes);
		SQLStmt.setIntAtName("series_id", ts.getId());
		SQLStmt.setLongAtName("exec_time", execTime);
		SQLStmt.setIntAtName("no_iter", iter);
		SQLStmt.setIntAtName("ref_no", ts.no_ref);
		SQLStmt.execute();
		SQLStmt.close();
	}

	/**
	 * main procedure that executes approximation
	 * 
	 * @throws NotMatchingSizeException
	 * @throws IndexOutOfMatrixBoundariesException
	 * @throws SQLException
	 */
	private void process0() throws NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException, SQLException {
		NO_OF_DIM_KEPT = mostCorrelated.size() > 2 ? ts.getCat02() : 1;// findRank(X.clone());

		long start = System.currentTimeMillis();
		SVD svd = new SVD(X);

		Matrix M;

		List<SVandIndex> svs = new ArrayList<IterativeSVD.SVandIndex>();
		for (int i = 0; i < svd.getS().colNum(); i++) {
			svs.add(new SVandIndex(i, svd.getS().getEntry(i, i)));
		}

		Comparator comp = Collections.reverseOrder();
		Collections.sort(svs, comp);

		M = new Matrix(X.rowNum(), X.colNum());
		for (int i = 0; i < NO_OF_DIM_KEPT; i++) {
			M = M.add(svd.getDimensionMatrix(svs.get(i).index));
		}

		double frobM = M.getFrobeniusNorm();

		double frobDelta = Double.MAX_VALUE;
		int k = 0;

		double frobM1 = frobM;
		Matrix M1;
		do {
			frobM = frobM1;

			for (int i = 0; i < ts_mvIndexes.length; i++) {
				X.setEntry(ts_mvIndexes[i] / ts.getGranularity(), 0,
						M.getEntry(ts_mvIndexes[i] / ts.getGranularity(), 0));
			}
			// save(); // TODO remove

			svd = new SVD(X);

			svs = new ArrayList<IterativeSVD.SVandIndex>();
			for (int i = 0; i < svd.getS().colNum(); i++) {
				svs.add(new SVandIndex(i, svd.getS().getEntry(i, i)));
			}

			Collections.sort(svs, comp);

			M1 = new Matrix(X.rowNum(), X.colNum());
			for (int i = 0; i < NO_OF_DIM_KEPT; i++) {
				M1 = M1.add(svd.getDimensionMatrix(svs.get(i).index));
			}

			frobM1 = M1.getFrobeniusNorm();

			frobDelta = M.subtract(M1).getFrobeniusNorm();
			k++;
			M = M1;
			logger.debug(k);
			if (k > max_iter)
				break;
		} while (frobDelta / frobM > threshold);

		save();
		// saveTime(System.currentTimeMillis() - start, k);

	}

	/**
	 * main procedure that executes approximation
	 * 
	 * @throws NotMatchingSizeException
	 * @throws IndexOutOfMatrixBoundariesException
	 * @throws SQLException
	 */
	private void process1() throws NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException, SQLException {
		NO_OF_DIM_KEPT = 1;

		SVD svd = new SVD(X);

		Matrix M;

		List<SVandIndex> svs = new ArrayList<IterativeSVD.SVandIndex>();
		for (int i = 0; i < svd.getS().colNum(); i++) {
			svs.add(new SVandIndex(i, svd.getS().getEntry(i, i)));
		}

		Comparator comp = Collections.reverseOrder();
		Collections.sort(svs, comp);

		M = new Matrix(X.rowNum(), X.colNum());
		for (int i = 0; i < NO_OF_DIM_KEPT; i++) {
			M = M.add(svd.getDimensionMatrix(svs.get(i).index));
		}

		double frobM = M.getFrobeniusNorm();

		double frobDelta = Double.MAX_VALUE;
		int k = 0;

		double frobM1 = frobM;
		Matrix M1;
		do {
			frobM = frobM1;

			for (int i = 0; i < ts_mvIndexes.length; i++) {
				X.setEntry(ts_mvIndexes[i] / ts.getGranularity(), 0,
						M.getEntry(ts_mvIndexes[i] / ts.getGranularity(), 0));
			}

			// save(); // TODO remove

			svd = new SVD(X);

			svs = new ArrayList<IterativeSVD.SVandIndex>();
			for (int i = 0; i < svd.getS().colNum(); i++) {
				svs.add(new SVandIndex(i, svd.getS().getEntry(i, i)));
			}

			Collections.sort(svs, comp);

			M1 = new Matrix(X.rowNum(), X.colNum());
			for (int i = 0; i < NO_OF_DIM_KEPT; i++) {
				M1 = M1.add(svd.getDimensionMatrix(svs.get(i).index));
			}

			frobM1 = M1.getFrobeniusNorm();

			frobDelta = M.subtract(M1).getFrobeniusNorm();
			k++;
			M = M1;
			logger.debug(k);
			if (k > max_iter)
				break;
		} while (frobDelta / frobM > threshold);
		// } while (k <= 100);// TODO: change
		save();

	}

	/**
	 * used for experiments only, applies basic recovery method for all time
	 * series that miss values
	 * 
	 * @throws SQLException
	 * @throws NotMatchingSizeException
	 * @throws InterpolationException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	public static void experimentStep3() throws SQLException,
			NotMatchingSizeException, InterpolationException,
			IndexOutOfMatrixBoundariesException {

		String query = "SELECT id,src_set,cat01,cat02 FROM series JOIN sets_series ON (src_series=id) WHERE cat01 IS NULL";

		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(query);

		OracleResultSet rs = (OracleResultSet) SQLStmt.executeQuery();
		while (rs.next()) {
			int series_id = rs.getInt("id");
			int set_id = rs.getInt("src_set");

			TimeSeries ts = new TimeSeries(series_id, new TSSet(set_id));
			logger.info(ts);
			ts.addMVBase4Methods();

		}

		SQLStmt.close();

		return;
	}

	/**
	 * used for experiments only, uses Iterative svd to recover missing values
	 * for all time series that miss values
	 * 
	 * @throws SQLException
	 * @throws NotMatchingSizeException
	 * @throws InterpolationException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	public static void experimentStep5() throws SQLException,
			NotMatchingSizeException, InterpolationException,
			IndexOutOfMatrixBoundariesException {
		String query = "SELECT id,src_set,cat01,cat02 FROM series JOIN sets_series ON (src_series=id) WHERE series_id >= 100 AND cat01 IS NOT NULL and align < 62 ORDER BY id";

		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(query);

		OracleResultSet rs = (OracleResultSet) SQLStmt.executeQuery();
		while (rs.next()) {
			int series_id = rs.getInt("id");
			int set_id = rs.getInt("src_set");

			TimeSeries ts = new TimeSeries(series_id, new TSSet(set_id));
			logger.info(ts);

			IterativeSVD svd = new IterativeSVD(ts);
		}

		SQLStmt.close();

		return;
	}

	public static void main(String[] args) throws NotMatchingSizeException,
			SQLException, InterpolationException,
			IndexOutOfMatrixBoundariesException {
		IterativeSVD.experimentStep3();
		// SVDNew.experimentStep5();

	}
}
