package ifi.dbtg.ldbe.algebra;


import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.Iterator;
import java.util.Random;

/**
 * represents matrix data type
 * @author Michal@Mourad
 *
 */
public class Matrix implements Iterable<Double> {
	public static String decimalFormat = " 0.000;-0.000";
	public static double LOWER_BOUND = 1E-10;

	private double[][] items;

	public Matrix(double[][] items) {
		this(items, false);
	}

	public Matrix(double[][] items, boolean refData) {
		if (refData)
			this.items = items;
		else {
			this.items = new double[items.length][items[0].length];
			for (int i = 0; i < items.length; i++) {
				System.arraycopy(items[i], 0, this.items[i], 0, items[i].length);
			}
		}
	}

	static public Matrix getRandMatrix(int m, int n) {
		Matrix res = new Matrix(m, n);
		Random r = new Random();

		for (int i = 0; i < m; i++)
			for (int j = 0; j < n; j++)
				res.setEntry(i, j, r.nextDouble());

		return res;
	}

	static public Matrix getEyeMatrix(int m, int n) {
		Matrix res = new Matrix(m, n);

		for (int i = 0; i < Math.min(m, n); i++)
			res.setEntry(i, i, 1.0);

		return res;
	}

	public double getFrobeniusNorm() {
		double sum = 0.0;
		for (double d : this) {
			sum += d * d;
		}
		return Math.sqrt(sum);
	}

	public Matrix(Object[] objects) {
		this.items = new double[objects.length][((Object[]) objects[0]).length - 1];
		for (int i = 0; i < items.length; i++) {
			for (int j = 0; j < items[0].length; j++) {
				// System.out.println ("i="+i+" j="+j);
				items[i][j] = ((BigDecimal) ((Object[]) objects[i])[j + 1])
						.doubleValue();
			}
		}

	}

	public Matrix(int rSize, int cSize) {
		this.items = new double[rSize][cSize];
	}

	public Matrix(Double[][] items) {
		this.items = new double[items.length][items[0].length];
		for (int i = 0; i < items.length; i++) {
			for (int j = 0; j < items[i].length; j++)
				this.items[i][j] = items[i][j];
		}
	}

	public Matrix clone() {
		return new Matrix(this.getData());
	}

	public static Matrix getDiagonalMatrix(double[] diagValues) {
		Matrix aux = new Matrix(diagValues.length, diagValues.length);
		for (int i = 0; i < diagValues.length; i++)
			aux.items[i][i] = diagValues[i];
		return aux;
	}

	public Iterator<Double> horizontalIterator() {
		return new Iterator<Double>() {
			int rowIndex = 0;
			int colIndex = 0;

			@Override
			public boolean hasNext() {
				return this.rowIndex < rowNum() - 1
						|| (this.rowIndex == rowNum() && this.colIndex < colNum());
			}

			public Double next() {
				double aux = getEntry(rowIndex, colIndex);
				colIndex++;
				if (colIndex == colNum()) {
					colIndex = 0;
					rowIndex++;
				}
				return aux;
			}

			@Override
			public void remove() throws UnsupportedOperationException {
				throw new UnsupportedOperationException("Unsupported opeartion");
			}
		};
	}

	public double getEntry(int i, int j) {
		if (i > this.rowNum() || j > this.colNum())
			throw new IndexOutOfBoundsException(
					"The index(es) of element out of matrix bounds");
		return this.items[i][j];
	}

	public void setEntry(int i, int j, double value) {
		if (i > this.rowNum() || j > this.colNum())
			throw new IndexOutOfBoundsException(
					"The index(es) of element out of matrix bounds");
		this.items[i][j] = value;
	}

	public Matrix add(Matrix m) throws NotMatchingSizeException {
		if (this.colNum() != m.colNum() || this.rowNum() != m.rowNum())
			throw new NotMatchingSizeException(
					"The dimensions of matrices differ!");
		double aux[][] = this.getData();

		for (int r = 0; r < this.rowNum(); r++)
			for (int c = 0; c < this.colNum(); c++)
				aux[r][c] += m.items[r][c];

		return new Matrix(aux, true);
	}

	public Matrix cloneColumn(int colNo) {
		double[][] aux = new double[this.rowNum()][this.colNum() + 1];
		for (int i = 0; i < this.rowNum(); i++) {
			for (int j = 0; j < this.colNum(); j++) {
				aux[i][j] = this.getEntry(i, j);
			}
			aux[i][this.colNum()] = this.getEntry(i, colNo);
		}
		return new Matrix(aux, true);
	}

	public Matrix subtract(Matrix m) throws NotMatchingSizeException {
		if (this.colNum() != m.colNum() || this.rowNum() != m.rowNum())
			throw new NotMatchingSizeException(
					"The dimensions of matrices differ!");
		double aux[][] = this.getData();

		for (int r = 0; r < this.rowNum(); r++)
			for (int c = 0; c < this.colNum(); c++)
				aux[r][c] -= m.items[r][c];

		return new Matrix(aux, true);
	}

	public Matrix multiply(Matrix m) throws NotMatchingSizeException {
		if (this.colNum() != m.rowNum())
			throw new NotMatchingSizeException(
					"Operation illegal for dimensions of matrices "
							+ this.rowNum() + "x" + this.colNum() + " and "
							+ +m.rowNum() + "x" + m.colNum());
		double aux[][] = new double[this.rowNum()][m.colNum()];

		for (int r = 0; r < this.rowNum(); r++)
			for (int c = 0; c < m.colNum(); c++) {
				double sum = 0.0;
				for (int i = 0; i < this.colNum(); i++)
					sum += this.items[r][i] * m.items[i][c];
				aux[r][c] = sum;
			}

		return new Matrix(aux, true);
	}

	public int colNum() {
		return this.items[0].length;
	}

	public int rowNum() {
		return this.items.length;
	}

	public String toString(boolean print) {
		DecimalFormat df = new DecimalFormat(decimalFormat);

		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < this.rowNum(); i++) {
			for (int j = 0; j < this.colNum(); j++) {
				sb.append(df.format(items[i][j]) + " & ");
			}
			sb.deleteCharAt(sb.length() - 2).append("\\\\\n");
		}
		sb.deleteCharAt(sb.length() - 1);
		return sb.toString();
	}

	public double[][] getDataRef() {
		return this.items;
	}

	public double[][] getData() {
		double[][] aux = new double[this.rowNum()][this.colNum()];
		for (int i = 0; i < this.rowNum(); i++) {
			System.arraycopy(this.items[i], 0, aux[i], 0, this.colNum());
		}
		return aux;
	}

	public double[] getColumn(int colIndex)
			throws IndexOutOfMatrixBoundariesException {
		if (colIndex >= this.colNum() || colIndex < 0)
			throw new IndexOutOfMatrixBoundariesException(
					"invalid column index");
		double[] aux = new double[this.rowNum()];
		for (int i = 0; i < this.rowNum(); i++) {
			aux[i] = this.items[i][colIndex];
		}
		return aux;
	}

	public double[] getRow(int rowIndex)
			throws IndexOutOfMatrixBoundariesException {
		if (rowIndex >= this.rowNum() || rowIndex < 0)
			throw new IndexOutOfMatrixBoundariesException("invalid row index");

		return items[rowIndex].clone();
	}

	public Matrix transpose() {
		double[][] data = new double[this.colNum()][this.rowNum()];

		for (int i = 0; i < this.rowNum(); i++) {
			for (int j = 0; j < this.colNum(); j++) {
				data[j][i] = this.getEntry(i, j);
			}
		}
		return new Matrix(data, true);
	}



	public void setColumn(int colIndex, double[] vector)
			throws NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {
		if (colIndex >= this.colNum() || colIndex < 0)
			throw new IndexOutOfMatrixBoundariesException(
					"invalid column index");
		if (vector.length != this.rowNum())
			throw new NotMatchingSizeException(
					"the size of column vector and height of the matrix differ");

		for (int i = 0; i < this.rowNum(); i++) {
			this.items[i][colIndex] = vector[i];
		}
	}



	@Override
	public Iterator<Double> iterator() {
		return this.horizontalIterator();
	}



	public double[] multiply(double[] z) throws NotMatchingSizeException {
		if (this.colNum() != z.length)
			throw new NotMatchingSizeException(
					"Operation illegal for dimensions of matrices "
							+ this.rowNum() + "x" + this.colNum() + " and "
							+ "1" + "x" + z.length);
		double aux[] = new double[this.rowNum()];

		for (int r = 0; r < this.rowNum(); r++) {
			double sum = 0.0;
			for (int i = 0; i < this.colNum(); i++)
				sum += this.items[r][i] * z[i];
			aux[r] = sum;
		}

		return aux;
	}
}
