package ifi.dbtg.ldbe.ts;

import ifi.dbtg.ldbe.algebra.IndexOutOfMatrixBoundariesException;
import ifi.dbtg.ldbe.algebra.NotMatchingSizeException;
import ifi.dbtg.ldbe.db.DBConnection;
import ifi.dbtg.ldbe.misc.InterpolationException;
import ifi.dbtg.ldbe.svd.IterativeSVD;

import java.math.BigDecimal;
import java.sql.Array;
import java.sql.Connection;
import java.sql.SQLException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.OracleResultSet;
import oracle.jdbc.OracleTypes;
import oracle.jdbc.internal.OraclePreparedStatement;

import org.apache.commons.math.stat.descriptive.moment.Mean;
import org.apache.commons.math.stat.descriptive.moment.StandardDeviation;
import org.apache.log4j.Logger;
import org.jfree.data.xy.XYSeries;

/**
 * Time series class
 * @author Michal@Mourad
 *
 */
public class TimeSeries {
	private class Wrapper implements Comparable {
		Double pearson;
		TimeSeries ts;

		@Override
		public int compareTo(Object o) {
			return pearson.compareTo(((Wrapper) o).pearson);
		}

		public Wrapper(TimeSeries ts, Double pearson) {
			this.ts = ts;
			this.pearson = pearson;
		}
	}

	private final static long serialVersionUID = 1L;
	private final static Logger logger = Logger.getLogger(TimeSeries.class);

	private final static String SQLQueryObservations = "BEGIN pkg_sest.load_series(?,?); END;";
	private final static String SQLQueryMVIndexes = "BEGIN pkg_sest.load_mv_indexes(?,?); END;";
	private final static String SQLQueryMVs = "BEGIN pkg_sest.get_mvs(?,?,?,?); END;";
	private final static String SQLQueryRVs = "BEGIN pkg_sest.get_rvs(?,?,?); END;";

	private final static String SQLQueryMVBlocks = "BEGIN pkg_sest.load_mv_blocks(?,?); END;";

	// file: '2010.11.18 [05] SEST TS queries.sql' ; query: '01 - series info'
	private final static String SQLQueryTSInfo = "SELECT granul gran, norm, TO_NUMBER(TO_CHAR(dt_fst,'YYYYMMDDHH24MISS')) dt_fst, TO_NUMBER(TO_CHAR(dt_lst,'YYYYMMDDHH24MISS')) dt_lst, label01 label, src_pk,pkg_sest.get_length(id) len,align,cat01,cat02,cat04,has_rmv,no_ref FROM series WHERE id=:id";

	// file: '2010.11.18 [05] SEST TS queries.sql' ; query: '02 - observations'
	// MIN and AVG'
	private final static String SQLQueryOBSminANDavg = "SELECT MIN(val) v_min,AVG(val) v_avg FROM observation WHERE series_id=:id";

	// file: '2010.11.18 [05] SEST TS queries.sql' ; query: '03 - insert MV base
	// 4'
	private final static String SQLQueryMVinsertBase4 = "INSERT INTO missing_value(series_id,add_dt,dt,ts,norm,val01,val02,val03,val04,val05,val06) VALUES(:id,CURRENT_DATE,null,:ts,:norm,:v_min,:v_avg,:v_nn1,:v_int,:v_int,:v_int) ";

	private final static String SQLQueryMVupdateBase4 = "UPDATE missing_value SET val01=:v_min,val02=:v_avg,val03=:v_nn1,val04=:v_int,val05=:v_int,val06=:v_int WHERE series_id=:id AND norm=:norm AND ts=:ts ";

	// file: '2010.11.18 [05] SEST TS queries.sql' ; '04 - update SERIES
	// alignment'
	private final static String SQLQuerySeriesUpdateAlign = "UPDATE series SET align=:align WHERE id=:id";

	private final int id;
	private final TSSet set;
	private long baseDt; // date of 1st observation, format: YYYYHHMMHH24MISS
	private long lastDt; // date of last observation, format: YYYYHHMMHH24MISS
	private int granularity;
	private int length;
	private Integer alignmentCode;
	private int src_pk;
	private boolean hasRMV;

	private Integer cat01;
	private Integer cat02, cat04;

	private NormalizationMethod normMthd;
	private String label;

	private Double[] data; // all values of TS (timestamp = index*granularity)
	private Integer[] mvIndexes; // indexes in data containing missing values
									// (MV)
	private Integer[] mvBlocks;
	public int no_ref;

	public Integer[] getMVindexes() throws SQLException {
		if (mvIndexes == null)
			loadMVIndexes();
		return mvIndexes;
	}

	public Integer[] getMVBlocks() throws SQLException {
		if (mvBlocks == null)
			loadMVBlocks();
		return mvBlocks;
	}

	public TimeSeries(int id, TSSet set) throws SQLException {
		this.id = id;
		this.set = set;
		this.loadInfo();
	}

	public Object[] toObjectArray() {
		return new Object[] { this.id, this.getLabel(), this.getLength(),
				this.getGranularity(), this.normMthd };
	}

	public String toString() {
		return Arrays.toString(toStringArray());
	}

	public List<TimeSeries> getMostCorrel(RecoveryMethod mthd1,
			RecoveryMethod mthd2) throws SQLException, InterpolationException,
			NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {
		ArrayList<Wrapper> pearsons = new ArrayList<Wrapper>();

		long time_satrt = System.currentTimeMillis();

		for (TimeSeries ts2nd : set.getTSList()) {
			// TODO consider allowing all series
			if (ts2nd.id != this.id && ts2nd.baseDt == this.baseDt
					&& ts2nd.lastDt >= this.lastDt) {
				logger.info(ts2nd);
				pearsons.add(new Wrapper(ts2nd, TimeSeries.pearson(this,
						ts2nd, mthd1, mthd2)));
			}
		}

		Comparator comparator = Collections.reverseOrder();
		Collections.sort(pearsons, comparator);

		logger.debug((System.currentTimeMillis() - time_satrt) / 1000
				+ " seconds.");

		List<TimeSeries> result = new ArrayList<TimeSeries>(this.no_ref);
		// TODO: change
		// if (this.isAligned(RecoveryMethod.SVD)) {
		for (int i = 0; i < Math.min(this.no_ref, pearsons.size()); i++) {
			result.add(pearsons.get(i).ts);
		}
		// } else {
		// for (int i = 0; i < nTopBest-1; i++) {
		// result.add(pearsons.get(tmp.get(i)));
		// }
		// Collections.sort(tmp);
		// result.add(pearsons.get(tmp.get(0)));
		// }

		return result;
	}

	public List<TimeSeries> getMostCorrel(int nTopBest, RecoveryMethod mthd1,
			RecoveryMethod mthd2, int shift) throws SQLException,
			InterpolationException,  NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {
		HashMap<Double, TimeSeries> pearsons = new HashMap<Double, TimeSeries>();

		long time_satrt = System.currentTimeMillis();

		for (TimeSeries ts2nd : set.getTSList()) {
			// TODO consider allowing all series
			if (ts2nd.id != this.id && ts2nd.baseDt == this.baseDt
					&& ts2nd.lastDt >= this.lastDt) {
				pearsons.put(TimeSeries.pearson(this, ts2nd, mthd1, mthd2),
						ts2nd);
			}
		}

		List<Double> tmp = new ArrayList(pearsons.keySet());

		Comparator comparator = Collections.reverseOrder();
		Collections.sort(tmp, comparator);
		DecimalFormat df = new DecimalFormat(IterativeSVD.decimalFormat);

		for (double d : tmp) {
			logger.debug("pearson: " + pearsons.get(d) + "\t\t" + df.format(d));
			// logger.debug("100001 & "+pearsons.get(d) + " & " + df.format(d)+
			// " \\\\ \\hline" );

		}

		logger.debug((System.currentTimeMillis() - time_satrt) / 1000
				+ " seconds.");

		List<TimeSeries> result = new ArrayList<TimeSeries>(this.no_ref);

		for (int i = shift; i < this.no_ref + shift; i++) {
			result.add(pearsons.get(tmp.get(i)));
		}

		return result;
	}

	public void addMVBase4Methods() throws SQLException, InterpolationException {
		if (!isAligned(RecoveryMethod.MIN_EVENTS)) {
			if (data == null)
				loadObservations();
			if (mvIndexes == null)
				loadMVIndexes();
			if (mvIndexes.length > 0) {

				double[] minANDavg = getMinANDAvgValues();
				logger.debug(Arrays.toString(minANDavg));

				// starting point for MV imputation
				int prevIndex = mvIndexes[0] / getGranularity() - 1;
				boolean inMVblock = false;

				Connection conn = DBConnection.getConnection();
				conn.setAutoCommit(false);

				OraclePreparedStatement SQLStmt = (OraclePreparedStatement) conn
						.prepareStatement(this.hasRMV() ? SQLQueryMVupdateBase4
								: SQLQueryMVinsertBase4);
				SQLStmt.setIntAtName("id", getId());
				SQLStmt.setIntAtName("norm", normMthd.ordinal());

				// go through all data between 1st MV and the last MV
				for (int i = prevIndex; i <= mvIndexes[mvIndexes.length - 1]
						/ getGranularity() + 1; i++) {
					// logger.info(this);
					if (data[i] == null) {
						inMVblock = true;
					} else {
						inMVblock = false;
					}

					if (!inMVblock) {
						if (i - prevIndex > 1) {

							double middle = prevIndex + (i - prevIndex - 1)
									/ 2.0;

							// logger.info(prevIndex * getGranularity() + " : "
							// + i * getGranularity());
							// logger.info(i - prevIndex);
							if ((i - prevIndex - 1) % 2 == 1) {
								SQLStmt.setIntAtName(
										"ts",
										((prevIndex + (i - prevIndex) / 2) * getGranularity()));
								SQLStmt.setDoubleAtName("v_min", minANDavg[0]);
								SQLStmt.setDoubleAtName("v_avg", minANDavg[1]);
								SQLStmt.setDoubleAtName(
										"v_nn1",
										data[prevIndex] < data[i] ? data[prevIndex]
												: data[i]);
								SQLStmt.setDoubleAtName(
										"v_int",
										getInterp(
												prevIndex * getGranularity(),
												data[prevIndex],
												i * getGranularity(),
												data[i],
												(prevIndex + (i - prevIndex) / 2)
														* getGranularity()));
								SQLStmt.execute();

								// logger.info(((prevIndex + (i - prevIndex) /
								// 2) *
								// granularity)
								// + " - MIN(val1,val2)");
							}
							for (int j = prevIndex + 1; j <= (int) (Math
									.floor(middle)); j++) {
								SQLStmt.setIntAtName("ts", j * getGranularity());
								SQLStmt.setDoubleAtName("v_min", minANDavg[0]);
								SQLStmt.setDoubleAtName("v_avg", minANDavg[1]);
								SQLStmt.setDoubleAtName("v_nn1",
										data[prevIndex]);
								SQLStmt.setDoubleAtName(
										"v_int",
										getInterp(prevIndex * getGranularity(),
												data[prevIndex], i
														* getGranularity(),
												data[i], j * getGranularity()));
								SQLStmt.execute();

								// logger.info(j * granularity + " - "
								// + (prevIndex * granularity));
							}
							for (int j = (int) (Math.ceil(middle)) + 1; j < i; j++) {
								SQLStmt.setIntAtName("ts", j * getGranularity());
								SQLStmt.setDoubleAtName("v_min", minANDavg[0]);
								SQLStmt.setDoubleAtName("v_avg", minANDavg[1]);
								SQLStmt.setDoubleAtName("v_nn1", data[i]);
								SQLStmt.setDoubleAtName(
										"v_int",
										getInterp(prevIndex * getGranularity(),
												data[prevIndex], i
														* getGranularity(),
												data[i], j * getGranularity()));
								SQLStmt.execute();

								// logger.info(j * granularity + " - "
								// + (i * granularity));
							}

						}
						prevIndex = i;
					}
				}

				int alignmentCode = 0;

				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.MIN_EVENTS);
				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.MEAN_EVENTS);
				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.NN1_EVENTS);
				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.LIN_INTERP_EVENTS);

				updateAlign(conn, alignmentCode);
				conn.commit();
				conn.setAutoCommit(true);

				SQLStmt.close();

				this.setAlignmentCode(alignmentCode);
			} else {
				int alignmentCode = 0;

				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.MIN_EVENTS);
				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.MEAN_EVENTS);
				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.NN1_EVENTS);
				alignmentCode += RecoveryMethod
						.getDeterminator(RecoveryMethod.LIN_INTERP_EVENTS);

				updateAlign(DBConnection.getConnection(), alignmentCode);
			}
		}
	}

	/**
	 * Function checks if the series is aligned with @param mthd method
	 * 
	 * @param mthd
	 * @return
	 */
	public boolean isAligned(RecoveryMethod mthd) {
		return (getAlignmentCode() & RecoveryMethod.getDeterminator(mthd)) > 0;
	}

	private void loadObservations() throws SQLException {
		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryObservations);

		SQLStmt.setInt(1, getId());
		SQLStmt.registerOutParameter(2, OracleTypes.ARRAY, "TY_SERIES");

		SQLStmt.execute();

		Array simpleArray = SQLStmt.getArray(2);
		BigDecimal[] msgArray = (BigDecimal[]) simpleArray.getArray();
		data = new Double[msgArray.length];
		for (int i = 0; i < msgArray.length; i++) {
			data[i] = (msgArray[i] == null ? null : msgArray[i].doubleValue());
		}

		SQLStmt.close();
		return;
	}

	private void loadMVIndexes() throws SQLException {
		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryMVIndexes);

		SQLStmt.setInt(1, getId());
		SQLStmt.registerOutParameter(2, OracleTypes.ARRAY, "TY_MV_INDEXES");

		SQLStmt.execute();

		Array simpleArray = SQLStmt.getArray(2);
		BigDecimal[] msgArray = (BigDecimal[]) simpleArray.getArray();
		mvIndexes = new Integer[msgArray.length];
		for (int i = 0; i < msgArray.length; i++) {
			mvIndexes[i] = msgArray[i].intValue();
		}

		SQLStmt.close();
		return;
	}

	private void loadMVBlocks() throws SQLException {
		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryMVBlocks);

		SQLStmt.setInt(1, getId());
		SQLStmt.registerOutParameter(2, OracleTypes.ARRAY, "TY_MV_INDEXES");

		SQLStmt.execute();

		Array simpleArray = SQLStmt.getArray(2);
		BigDecimal[] msgArray = (BigDecimal[]) simpleArray.getArray();
		mvBlocks = new Integer[msgArray.length];

		for (int i = 0; i < msgArray.length; i++) {
			mvBlocks[i] = msgArray[i].intValue();
		}

		SQLStmt.close();
		return;
	}

	private void loadInfo() throws SQLException {
		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryTSInfo);

		SQLStmt.setInt(1, getId());

		OracleResultSet rs = (OracleResultSet) SQLStmt.executeQuery();
		rs.next();

		this.setLength(rs.getInt("len"));
		this.setGranularity(rs.getInt("gran"));
		this.setNormMthd(NormalizationMethod.getNormalization(rs.getInt("norm")));
		this.baseDt = rs.getLong("dt_fst");
		this.lastDt = rs.getLong("dt_lst");
		this.setLabel(rs.getString("label"));
		this.setAlignmentCode(rs.getInt("align"));
		this.setSrc_pk(rs.getInt("src_pk"));
		this.cat01 = rs.getInt("cat01");
		this.setCat02(rs.getInt("cat02"));
		this.setCat04(rs.getInt("cat04"));
		this.setHasRMV(rs.getInt("has_rmv") == 1);
		this.no_ref=rs.getInt("no_ref");

		// logger.info(baseDt);
		// logger.info(lastDt);

		if (rs.next())
			throw new RuntimeException("More rows than expected");

		SQLStmt.close();
		return;
	}

	public void updateAlign(Connection conn, int alignCode) throws SQLException {
		OracleCallableStatement SQLStmt = (OracleCallableStatement) conn
				.prepareCall(SQLQuerySeriesUpdateAlign);

		SQLStmt.setIntAtName("id", getId());
		SQLStmt.setIntAtName("align", alignCode);

		SQLStmt.execute();

		SQLStmt.close();
		return;
	}

	private double[] getMinANDAvgValues() throws SQLException {
		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryOBSminANDavg);

		SQLStmt.setInt(1, getId());

		OracleResultSet rs = (OracleResultSet) SQLStmt.executeQuery();
		rs.next();

		double min = rs.getDouble("v_min");
		double avg = rs.getDouble("v_avg");

		if (rs.next())
			throw new RuntimeException("More rows than expected");

		SQLStmt.close();

		return new double[] { min, avg };
	}

	public XYSeries getXYSeriesOriginal(long shift) throws SQLException {
		if (data == null)
			loadObservations();

		XYSeries series = new XYSeries(this.getLabel());
		for (int i = 0; i < data.length; i++) {
			series.add(i * getGranularity() + shift, data[i]);
		}

		return series;
	}

	private Object[] getMVs(long shift, int alignCode) throws SQLException,
			InterpolationException, NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {
		if ((alignCode & this.alignmentCode) != alignCode)
			addMVBase4Methods();

		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryMVs);

		SQLStmt.registerOutParameter(4, OracleTypes.ARRAY, "TY_MVS");
		SQLStmt.setInt(1, getId());
		SQLStmt.setInt(2, normMthd.ordinal());
		SQLStmt.setInt(3, alignCode);
		SQLStmt.execute();

		Array simpleArray = SQLStmt.getArray(4);
		Object[] msgArray = (Object[]) simpleArray.getArray();

		for (int i = 0; i < msgArray.length; i++) {
			msgArray[i] = (Object[]) ((Array) msgArray[i]).getArray();
		}

		SQLStmt.close();
		return msgArray;
	}

	private Object[] getRVs(long shift) throws SQLException,
			InterpolationException, NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {

		OracleCallableStatement SQLStmt = (OracleCallableStatement) DBConnection
				.getConnection().prepareCall(SQLQueryRVs);

		SQLStmt.registerOutParameter(3, OracleTypes.ARRAY, "TY_MVS");
		SQLStmt.setInt(1, getId());
		SQLStmt.setInt(2, normMthd.ordinal());
		SQLStmt.execute();

		Array simpleArray = SQLStmt.getArray(3);
		Object[] msgArray = (Object[]) simpleArray.getArray();

		for (int i = 0; i < msgArray.length; i++) {
			msgArray[i] = (Object[]) ((Array) msgArray[i]).getArray();
		}

		SQLStmt.close();
		return msgArray;
	}

	// uses original data to replace NULLs with values from imputation method
	public XYSeries getXYSeries(long shift, RecoveryMethod mthd)
			throws SQLException,  NotMatchingSizeException,
			InterpolationException, IndexOutOfMatrixBoundariesException {
		XYSeries series = getXYSeriesOriginal(shift);

		if (mthd == RecoveryMethod.DELETE_EVENTS) {
			for (Integer idx : getMVindexes())
				series.remove(idx);

		} else {
			Object[] mvs = getMVs(shift, RecoveryMethod.getDeterminator(mthd));

			for (Object dataRow : mvs) {
				series.updateByIndex(((BigDecimal[]) dataRow)[0].intValue()
						/ getGranularity(),
						((BigDecimal[]) dataRow)[mthd.ordinal()].doubleValue());
			}
		}
		return series;
	}

	// returns series with original removed data only
	public XYSeries getXYRVSeries(long shift) throws SQLException,
			NotMatchingSizeException, InterpolationException,
			IndexOutOfMatrixBoundariesException {

		XYSeries series = new XYSeries("Removed observations");

		for (int i = 0; i < data.length; i++) {
			series.add(i * getGranularity() + shift, data[i]);
		}

		Object[] mvs = getRVs(shift);

		for (Object dataRow : mvs) {
			series.updateByIndex(((BigDecimal[]) dataRow)[0].intValue()
					/ getGranularity(),
					((BigDecimal[]) dataRow)[1].doubleValue());
		}

		return series;
	}

	// gets new array in which the missing values are filled (the rest of
	// references is kept)
	public Double[] getFullData(long shift, RecoveryMethod mthd)
			throws SQLException, InterpolationException,
			NotMatchingSizeException, IndexOutOfMatrixBoundariesException {
		if (data == null)
			loadObservations();
		Double[] data_copy = data.clone();
		// logger.debug(this);
		if (!isAligned(mthd)) {
			addMVBase4Methods();
		}

		Object[] mvs = getMVs(shift, RecoveryMethod.getDeterminator(mthd));
		if (mvs.length > 0) {
			for (Object dataRow : mvs) {
				data_copy[((BigDecimal[]) dataRow)[0].intValue()
						/ getGranularity()] = ((BigDecimal[]) dataRow)[mthd
						.ordinal()].doubleValue();
			}
		}
		return data_copy;
	}

	public void setNormMthd(NormalizationMethod normMthd) {
		this.normMthd = normMthd;
	}

	public NormalizationMethod getNormMthd() {
		return normMthd;
	}

	public static double getInterp(double x0, double y0, double x1, double y1,
			double x2) throws InterpolationException {
		double aux = y0 + (y1 - y0) / (x1 - x0) * (x2 - x0);
		if (Double.isInfinite(aux))
			throw new InterpolationException();
		return aux;
	}



	/**
	 *  optimized version of pearson
	 * @param ts
	 * @param candTS
	 * @param method1
	 * @param method2
	 * @return
	 * @throws SQLException
	 * @throws InterpolationException
	 * @throws NotMatchingSizeException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	
	public static double pearson(TimeSeries ts, TimeSeries candTS,
			RecoveryMethod method1, RecoveryMethod method2)
			throws SQLException, InterpolationException, 
			NotMatchingSizeException, IndexOutOfMatrixBoundariesException {
		double pearson = 0.0;

		if (ts.data == null)
			ts.loadObservations();
		if (candTS.data == null)
			candTS.loadObservations();

		// TODO optimization:
		// http://www.vias.org/tmdatanaleng/cc_corr_coeff.html

		Double[] dataTS = ts.getFullData(0, method1);
		Double[] dataCandTS = candTS.getFullData(0, method2);

		double x = 0, y = 0;
		double xx = 0, yy = 0;
		double xy = 0;

		for (int i = 0; i < dataTS.length; i++) {
			// logger.info(i);
			x += dataTS[i];
			xx += dataTS[i] * dataTS[i];

			y += dataCandTS[i];
			yy += dataCandTS[i] * dataCandTS[i];

			xy += dataTS[i] * dataCandTS[i];

		}

		pearson = ((xy) - (x * y / dataTS.length))
				/ (Math.sqrt(xx - x * x / dataTS.length) * Math.sqrt(yy - y * y
						/ dataTS.length));
		return Math.abs(pearson);

	}

	public static void main(String[] args) throws SQLException,
			InterpolationException,  NotMatchingSizeException,
			IndexOutOfMatrixBoundariesException {
	}

	public Integer getId() {
		return id;
	}

	public void setLength(int length) {
		this.length = length;
	}

	public Integer getLength() {
		return length;
	}

	public void setGranularity(int granularity) {
		this.granularity = granularity;
	}

	public Integer getGranularity() {
		return granularity;
	}

	public void setAlignmentCode(Integer alignmentCode) {
		this.alignmentCode = alignmentCode;
	}

	public Integer getAlignmentCode() {
		return alignmentCode;
	}

	public void setLabel(String label) {
		this.label = label;
	}

	public String getLabel() {
		return label;
	}

	public void setSrc_pk(int src_pk) {
		this.src_pk = src_pk;
	}

	public int getSrc_pk() {
		return src_pk;
	}

	public String[] toStringArray() {
		return new String[] { this.getId().toString(), this.getLabel(),
				this.getLength().toString(), this.getGranularity().toString(),
				this.normMthd.toString() };
	}

	public void setCat02(Integer cat02) {
		this.cat02 = cat02;
	}

	public Integer getCat02() {
		return cat02;
	}

	public void setCat04(Integer cat04) {
		this.cat04 = cat04;
	}

	public Integer getCat04() {
		return cat04;
	}

	public void setHasRMV(boolean hasRMV) {
		this.hasRMV = hasRMV;
	}

	public boolean hasRMV() {
		return hasRMV;
	}
}
