package ifi.dbtg.ldbe.charts;

import ifi.dbtg.ldbe.ts.RecoveryMethod;
import ifi.dbtg.ldbe.algebra.IndexOutOfMatrixBoundariesException;
import ifi.dbtg.ldbe.algebra.NotMatchingSizeException;
import ifi.dbtg.ldbe.misc.InterpolationException;
import ifi.dbtg.ldbe.svd.IterativeSVD;
import ifi.dbtg.ldbe.ts.TimeSeries;

import org.jfree.chart.ChartColor;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.JFrame;

import org.apache.log4j.Logger;
import org.jfree.chart.ChartFrame;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberAxis3D;
import org.jfree.chart.axis.TickUnitSource;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.XYSeriesCollection;

/**
 * Handles displaying of recovered missing values and if missing values not
 * recovered call recovery methods
 * 
 * @author Michal@Mourad
 * 
 */
public class ChartsManager extends JFrame {
	final static Logger logger = Logger.getLogger(ChartsManager.class);

	private static final long serialVersionUID = 1L;
	private List<ChartFrame> charts;
	static int sideWindowsSizeWidth = 400;
	static int sideWindowsSizeHeight = 300;

	/**
	 * creates window containing recovery for particular recovery method
	 * 
	 * @param mthd
	 *            recovery method
	 * @param ts
	 *            time series for which recovery was performed
	 * @throws Exception
	 */
	private void newChart(RecoveryMethod mthd, TimeSeries ts) throws Exception {

		NumberAxis xAxis = new NumberAxis3D("Value");
		NumberAxis yAxis = new NumberAxis3D("Relative timestamp");

		TickUnitSource units = NumberAxis.createIntegerTickUnits();
		yAxis.setStandardTickUnits(units);

		XYLineAndShapeRenderer renderer = new XYLineAndShapeRenderer(true,
				false);

		if (mthd == RecoveryMethod.SVD && !ts.isAligned(mthd)) {
			new IterativeSVD(ts);
		}

		

		renderer.setSeriesPaint(0, ChartColor.BLACK);
		renderer.setSeriesPaint(1, ChartColor.RED);
		renderer.setSeriesPaint(2, ChartColor.BLUE);
		renderer.setSeriesPaint(3, ChartColor.DARK_GREEN);
		renderer.setSeriesPaint(4, ChartColor.DARK_MAGENTA);

		XYSeriesCollection dataset = getXYSeriesCollection(ts, mthd);

		ScrollableXYPlot xyPlot = new ScrollableXYPlot(dataset, yAxis, xAxis,
				renderer, ts.getLength() * ts.getGranularity());

		JFreeChart chart = new JFreeChart("Recovered Time Series",
				JFreeChart.DEFAULT_TITLE_FONT, xyPlot, true);
		xyPlot.setRangePannable(false);
		xyPlot.setDomainPannable(false);

		ChartFrame frame = new ChartFrame(mthd.name(), chart);

		frame.pack();
		frame.setVisible(true);
		frame.setSize(ChartsManager.sideWindowsSizeWidth,
				ChartsManager.sideWindowsSizeHeight);
		frame.setLocation((this.charts.size() % 3)
				* ChartsManager.sideWindowsSizeWidth, (this.charts.size() / 3)
				* ChartsManager.sideWindowsSizeHeight);

		this.charts.add(frame);
	}

	/**
	 * retrieves time series data to display in chart
	 * 
	 * @param ts
	 *            time series
	 * @return time series data for chart
	 * @throws SQLException
	 * @throws IterationError
	 * @throws NotMatchingSizeException
	 * @throws InterpolationException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	public XYSeriesCollection getXYSeriesCollection(TimeSeries ts)
			throws SQLException, NotMatchingSizeException,
			InterpolationException, IndexOutOfMatrixBoundariesException {
		XYSeriesCollection collection = new XYSeriesCollection();
		collection.addSeries(ts.getXYSeriesOriginal(0));

		return collection;
	}

	/**
	 * retrieves time series data with missing values recovered by given method
	 * to display in chart
	 * 
	 * @param ts
	 *            time series
	 * @return time series data for chart
	 * 
	 * @param mthd
	 *            recovery method
	 * @return
	 * @throws SQLException
	 * @throws IterationError
	 * @throws NotMatchingSizeException
	 * @throws InterpolationException
	 * @throws IndexOutOfMatrixBoundariesException
	 */
	public XYSeriesCollection getXYSeriesCollection(TimeSeries ts,
			RecoveryMethod mthd) throws SQLException,
			NotMatchingSizeException, InterpolationException,
			IndexOutOfMatrixBoundariesException {
		XYSeriesCollection collection = new XYSeriesCollection();
		collection.addSeries(ts.getXYSeries(0, mthd));

		if (mthd == RecoveryMethod.SVD) {
			for (TimeSeries ts_cor : ts.getMostCorrel(RecoveryMethod.SVD,
					RecoveryMethod.LIN_INTERP_EVENTS)) {
				collection.addSeries(ts_cor.getXYSeries(0,
						RecoveryMethod.LIN_INTERP_EVENTS));
			}
		}
		return collection;
	}

	/**
	 * displays original time series data and calls displaying of recovered data
	 * 
	 * @param ts
	 *            time series
	 * @param align_mthds
	 *            list of methods for which recoveries should be also displayed
	 * @throws Exception
	 */
	public ChartsManager(TimeSeries ts, List<RecoveryMethod> align_mthds)
			throws Exception {
		super();

		this.charts = new ArrayList<ChartFrame>();
		NumberAxis xAxis = new NumberAxis3D("Value");
		NumberAxis yAxis = new NumberAxis3D("Relative timestamp");

		TickUnitSource units = NumberAxis.createIntegerTickUnits();
		yAxis.setStandardTickUnits(units);

		XYLineAndShapeRenderer renderer = new XYLineAndShapeRenderer(true, true);
		renderer.setSeriesPaint(0, ChartColor.BLACK);
		ScrollableXYPlot xyPlot = new ScrollableXYPlot(
				getXYSeriesCollection(ts), yAxis, xAxis, renderer,
				ts.getLength() * ts.getGranularity());

		xyPlot.setMVBlocksList(Arrays.asList(ts.getMVBlocks()));
		JFreeChart chart = new JFreeChart("Original Time Series",
				JFreeChart.DEFAULT_TITLE_FONT, xyPlot, true);

		for (RecoveryMethod mthd : align_mthds) {
			this.newChart(mthd, ts);
		}

		ChartPanel chartPanel = new ChartPanel(chart);

		try {
			ScrollableChartPane scrollableChartPane = new ScrollableChartPane(
					chartPanel, charts);
			this.setContentPane(scrollableChartPane);
			setSize(ChartsManager.sideWindowsSizeWidth * 2,
					ChartsManager.sideWindowsSizeHeight * 2);
			setLocation(600, 0);
			setVisible(true);
			setResizable(false);
			addWindowListener(new CloseListener());

		} catch (SQLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * listener to window closing event
	 * 
	 * @author Michal@Mourad
	 * 
	 */
	private class CloseListener extends WindowAdapter {
		public void windowClosing(WindowEvent event) {
			dispose();
			// System.exit(0);
		}
	}

	/**
	 * window disposer
	 */
	@Override
	public void dispose() {

		for (ChartFrame f : this.charts)
			f.dispose();

		super.dispose();
	}
}